/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.common.security.ssl;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Paths;
import java.security.GeneralSecurityException;
import java.security.KeyStore;
import java.security.SecureRandom;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;
import javax.net.ssl.KeyManager;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLEngine;
import javax.net.ssl.SSLParameters;
import javax.net.ssl.TrustManagerFactory;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.config.SslClientAuth;
import org.apache.kafka.common.config.types.Password;
import org.apache.kafka.common.network.Mode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SslEngineBuilder {
    private static final Logger log = LoggerFactory.getLogger(SslEngineBuilder.class);
    private final Map<String, ?> configs;
    private final String protocol;
    private final String provider;
    private final String kmfAlgorithm;
    private final String tmfAlgorithm;
    private final SecurityStore keystore;
    private final SecurityStore truststore;
    private final String[] cipherSuites;
    private final String[] enabledProtocols;
    private final SecureRandom secureRandomImplementation;
    private final SSLContext sslContext;
    private final SslClientAuth sslClientAuth;

    SslEngineBuilder(Map<String, ?> configs) {
        this.configs = Collections.unmodifiableMap(configs);
        this.protocol = (String)configs.get("ssl.protocol");
        this.provider = (String)configs.get("ssl.provider");
        List cipherSuitesList = (List)configs.get("ssl.cipher.suites");
        this.cipherSuites = cipherSuitesList != null && !cipherSuitesList.isEmpty() ? cipherSuitesList.toArray(new String[cipherSuitesList.size()]) : null;
        List enabledProtocolsList = (List)configs.get("ssl.enabled.protocols");
        this.enabledProtocols = enabledProtocolsList != null && !enabledProtocolsList.isEmpty() ? enabledProtocolsList.toArray(new String[enabledProtocolsList.size()]) : null;
        this.secureRandomImplementation = SslEngineBuilder.createSecureRandom((String)configs.get("ssl.secure.random.implementation"));
        this.sslClientAuth = SslEngineBuilder.createSslClientAuth((String)configs.get("ssl.client.auth"));
        this.kmfAlgorithm = (String)configs.get("ssl.keymanager.algorithm");
        this.tmfAlgorithm = (String)configs.get("ssl.trustmanager.algorithm");
        this.keystore = SslEngineBuilder.createKeystore((String)configs.get("ssl.keystore.type"), (String)configs.get("ssl.keystore.location"), (Password)configs.get("ssl.keystore.password"), (Password)configs.get("ssl.key.password"));
        this.truststore = SslEngineBuilder.createTruststore((String)configs.get("ssl.truststore.type"), (String)configs.get("ssl.truststore.location"), (Password)configs.get("ssl.truststore.password"));
        this.sslContext = this.createSSLContext();
    }

    private static SslClientAuth createSslClientAuth(String key) {
        SslClientAuth auth = SslClientAuth.forConfig(key);
        if (auth != null) {
            return auth;
        }
        log.warn("Unrecognized client authentication configuration {}.  Falling back to NONE.  Recognized client authentication configurations are {}.", (Object)key, (Object)String.join((CharSequence)", ", SslClientAuth.VALUES.stream().map(a -> a.name()).collect(Collectors.toList())));
        return SslClientAuth.NONE;
    }

    private static SecureRandom createSecureRandom(String key) {
        if (key == null) {
            return null;
        }
        try {
            return SecureRandom.getInstance(key);
        }
        catch (GeneralSecurityException e) {
            throw new KafkaException(e);
        }
    }

    private SSLContext createSSLContext() {
        try {
            SSLContext sslContext = this.provider != null ? SSLContext.getInstance(this.protocol, this.provider) : SSLContext.getInstance(this.protocol);
            KeyManager[] keyManagers = null;
            if (this.keystore != null || this.kmfAlgorithm != null) {
                String kmfAlgorithm = this.kmfAlgorithm != null ? this.kmfAlgorithm : KeyManagerFactory.getDefaultAlgorithm();
                KeyManagerFactory kmf = KeyManagerFactory.getInstance(kmfAlgorithm);
                if (this.keystore != null) {
                    KeyStore ks = this.keystore.load();
                    Password keyPassword = this.keystore.keyPassword != null ? this.keystore.keyPassword : this.keystore.password;
                    kmf.init(ks, keyPassword.value().toCharArray());
                } else {
                    kmf.init(null, null);
                }
                keyManagers = kmf.getKeyManagers();
            }
            String tmfAlgorithm = this.tmfAlgorithm != null ? this.tmfAlgorithm : TrustManagerFactory.getDefaultAlgorithm();
            TrustManagerFactory tmf = TrustManagerFactory.getInstance(tmfAlgorithm);
            KeyStore ts = this.truststore == null ? null : this.truststore.load();
            tmf.init(ts);
            sslContext.init(keyManagers, tmf.getTrustManagers(), this.secureRandomImplementation);
            log.debug("Created SSL context with keystore {}, truststore {}", (Object)this.keystore, (Object)this.truststore);
            return sslContext;
        }
        catch (Exception e) {
            throw new KafkaException(e);
        }
    }

    private static SecurityStore createKeystore(String type, String path, Password password, Password keyPassword) {
        if (path == null && password != null) {
            throw new KafkaException("SSL key store is not specified, but key store password is specified.");
        }
        if (path != null && password == null) {
            throw new KafkaException("SSL key store is specified, but key store password is not specified.");
        }
        if (path != null && password != null) {
            return new SecurityStore(type, path, password, keyPassword);
        }
        return null;
    }

    private static SecurityStore createTruststore(String type, String path, Password password) {
        if (path == null && password != null) {
            throw new KafkaException("SSL trust store is not specified, but trust store password is specified.");
        }
        if (path != null) {
            return new SecurityStore(type, path, password, null);
        }
        return null;
    }

    Map<String, Object> configs() {
        return this.configs;
    }

    public SecurityStore keystore() {
        return this.keystore;
    }

    public SecurityStore truststore() {
        return this.truststore;
    }

    public SSLEngine createSslEngine(Mode mode, String peerHost, int peerPort, String endpointIdentification) {
        SSLEngine sslEngine = this.sslContext.createSSLEngine(peerHost, peerPort);
        if (this.cipherSuites != null) {
            sslEngine.setEnabledCipherSuites(this.cipherSuites);
        }
        if (this.enabledProtocols != null) {
            sslEngine.setEnabledProtocols(this.enabledProtocols);
        }
        if (mode == Mode.SERVER) {
            sslEngine.setUseClientMode(false);
            switch (this.sslClientAuth) {
                case REQUIRED: {
                    sslEngine.setNeedClientAuth(true);
                    break;
                }
                case REQUESTED: {
                    sslEngine.setWantClientAuth(true);
                    break;
                }
            }
            sslEngine.setUseClientMode(false);
        } else {
            sslEngine.setUseClientMode(true);
            SSLParameters sslParams = sslEngine.getSSLParameters();
            sslParams.setEndpointIdentificationAlgorithm(endpointIdentification);
            sslEngine.setSSLParameters(sslParams);
        }
        return sslEngine;
    }

    public SSLContext sslContext() {
        return this.sslContext;
    }

    public boolean shouldBeRebuilt(Map<String, Object> nextConfigs) {
        if (!nextConfigs.equals(this.configs)) {
            return true;
        }
        if (this.truststore != null && this.truststore.modified()) {
            return true;
        }
        return this.keystore != null && this.keystore.modified();
    }

    static class SecurityStore {
        private final String type;
        private final String path;
        private final Password password;
        private final Password keyPassword;
        private final Long fileLastModifiedMs;

        SecurityStore(String type, String path, Password password, Password keyPassword) {
            Objects.requireNonNull(type, "type must not be null");
            this.type = type;
            this.path = path;
            this.password = password;
            this.keyPassword = keyPassword;
            this.fileLastModifiedMs = this.lastModifiedMs(path);
        }

        /*
         * Enabled aggressive block sorting
         * Enabled unnecessary exception pruning
         * Enabled aggressive exception aggregation
         */
        KeyStore load() {
            try (InputStream in = Files.newInputStream(Paths.get(this.path, new String[0]), new OpenOption[0]);){
                KeyStore ks = KeyStore.getInstance(this.type);
                char[] passwordChars = this.password != null ? this.password.value().toCharArray() : null;
                ks.load(in, passwordChars);
                KeyStore keyStore = ks;
                return keyStore;
            }
            catch (IOException | GeneralSecurityException e) {
                throw new KafkaException("Failed to load SSL keystore " + this.path + " of type " + this.type, e);
            }
        }

        private Long lastModifiedMs(String path) {
            try {
                return Files.getLastModifiedTime(Paths.get(path, new String[0]), new LinkOption[0]).toMillis();
            }
            catch (IOException e) {
                log.error("Modification time of key store could not be obtained: " + path, e);
                return null;
            }
        }

        boolean modified() {
            Long modifiedMs = this.lastModifiedMs(this.path);
            return modifiedMs != null && !Objects.equals(modifiedMs, this.fileLastModifiedMs);
        }

        public String toString() {
            return "SecurityStore(path=" + this.path + ", modificationTime=" + (this.fileLastModifiedMs == null ? null : new Date(this.fileLastModifiedMs)) + ")";
        }
    }
}

