package com.chinamcloud.common.storage.util;

import com.alibaba.fastjson.JSON;
import com.aliyun.oss.ClientException;
import com.aliyun.oss.OSSClient;
import com.aliyun.oss.OSSException;
import com.aliyun.oss.model.Bucket;
import com.aliyun.oss.model.CannedAccessControlList;
import com.aliyun.oss.model.CreateBucketRequest;
import com.chinamcloud.common.result.ResultDTO;
import com.chinamcloud.common.storage.dto.BucketBaseDTO;
import com.chinamcloud.common.storage.dto.BucketDTO;
import com.chinamcloud.common.storage.transformer.Bucket2DTOTransformer;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;

/**
 * Created with IntelliJ IDEA.
 * User: chenzhiwei
 * Date: 17/1/9
 * Time: 下午3:03
 */
@Slf4j
public class ThirdStorageBucketUtil {
    /**
     * 创建bucket
     * @param bucketBaseDTO
     * @return
     */
    public static ResultDTO<BucketDTO> createBucket(BucketBaseDTO bucketBaseDTO){
        ResultDTO<Boolean> booleanResultDTO = checkBucketBaseDTO(bucketBaseDTO);
        if (!booleanResultDTO.isSuccess()) {
            return ResultDTO.fail(booleanResultDTO.getDescription());
        }
        ResultDTO<Boolean> queryIsExistBucketResultDTO = queryIsExistBucket(bucketBaseDTO);
        if(queryIsExistBucketResultDTO.isSuccess() && queryIsExistBucketResultDTO.getData()){
            return ResultDTO.fail("'"+bucketBaseDTO.getBucketName()+"'名称的存储空间已存在，请创建其他名称的存储空间");
        }
        OSSClient ossClient = OSSClientUtil.createOssClient(bucketBaseDTO.getEndpoint());
        CreateBucketRequest createBucketRequest = new CreateBucketRequest(bucketBaseDTO.getBucketName());
        createBucketRequest.setCannedACL(CannedAccessControlList.parse(bucketBaseDTO.getCannedAccessControlList().name()));
        try {
            Bucket bucket = ossClient.createBucket(createBucketRequest);
            log.info("bucket:{}", JSON.toJSONString(bucket,true));
            BucketDTO bucketDTO = Bucket2DTOTransformer.INSTANCE.apply(bucket);
            return ResultDTO.successfy(bucketDTO);
        }catch (OSSException ossException){
            log.error("创建存储空间遇到OSSException异常",ossException);
            return ResultDTO.fail("创建存储空间遇到异常:"+ossException.getMessage());
        }catch (ClientException e){
            log.error("创建存储空间遇到ClientException异常",e);
            return ResultDTO.fail("创建存储空间遇到客户端异常");
        }finally {
            ossClient.shutdown();
        }
    }

    /**
     * 删除bucket
     * @param bucketBaseDTO
     * @return true:删除成功，false：删除失败
     */
    public static ResultDTO<Boolean> deleteBucket(BucketBaseDTO bucketBaseDTO){
        ResultDTO<Boolean> booleanResultDTO = checkBucketBaseDTO(bucketBaseDTO);
        if (!booleanResultDTO.isSuccess()) {
            return booleanResultDTO;
        }
        OSSClient ossClient = OSSClientUtil.createOssClient(bucketBaseDTO.getEndpoint());
        try {
            ossClient.deleteBucket(bucketBaseDTO.getBucketName());
            log.info("bucketBaseDTO:{} is delete",bucketBaseDTO);
            return ResultDTO.successfy(true);
        }catch (OSSException ossException){
            log.error("删除存储空间遇到OSSException异常",ossException);
            return ResultDTO.fail("删除存储空间遇到异常:"+ossException.getMessage());
        }catch (ClientException e){
            log.error("删除存储空间遇到ClientException异常",e);
            return ResultDTO.fail("删除存储空间遇到客户端异常");
        }finally {
            ossClient.shutdown();
        }
    }

    private static ResultDTO<Boolean> checkBucketBaseDTO(BucketBaseDTO bucketBaseDTO) {
        if(bucketBaseDTO == null ){
            return ResultDTO.fail("bucketCreateDTO is not null");
        }
        String bucketName = bucketBaseDTO.getBucketName();
        if(StringUtils.isBlank(bucketName)){
            return ResultDTO.fail("bucketName is not null");
        }
        if(bucketName.length()<3 || bucketName.length()>63){
            return ResultDTO.fail("3 <= bucketName.length < 63");
        }
        return ResultDTO.successfy(true);
    }

    /**
     * 判断bucket是否存在：
     * 首先要判断业务是否正常处理（ResultDTO.isSuccess（）），判断为true后，再判断返回的Boolean是true还是false.
     * @param bucketBaseDTO
     * @return true：存在，false:不存在
     */
    public static ResultDTO<Boolean> isExistBucket(BucketBaseDTO bucketBaseDTO){
        ResultDTO<Boolean> booleanResultDTO = checkBucketBaseDTO(bucketBaseDTO);
        if (!booleanResultDTO.isSuccess()) {
            return booleanResultDTO;
        }
        return queryIsExistBucket(bucketBaseDTO);
    }

    private static ResultDTO<Boolean> queryIsExistBucket(BucketBaseDTO bucketBaseDTO) {
        OSSClient ossClient = OSSClientUtil.createOssClient(bucketBaseDTO.getEndpoint());
        try {
            boolean isExist = ossClient.doesBucketExist(bucketBaseDTO.getBucketName());
            return ResultDTO.successfy(isExist);
        }catch (OSSException ossException){
            log.error("查询存储空间是否存在遇到OSSException异常",ossException);
            return ResultDTO.fail("查询存储空间是否存在遇到异常:"+ossException.getMessage());
        }catch (ClientException e){
            log.error("查询存储空间是否存在遇到ClientException异常",e);
            return ResultDTO.fail("查询存储空间是否存在遇到客户端异常");
        }finally {
            ossClient.shutdown();
        }
    }

}
