package com.chinamcloud.common.storage.util;

import com.chinamcloud.common.result.ResultDTO;
import com.chinamcloud.common.storage.dto.*;
import com.chinamcloud.common.util.FileUtil;
import com.google.common.collect.Lists;
import com.google.common.math.IntMath;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.fileupload.FileItem;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.math.RoundingMode;
import java.util.List;

/**
 * Created with IntelliJ IDEA.
 * User: chenzhiwei
 * Date: 17/2/6
 * Time: 下午2:01
 */
@Slf4j
public class FileStorageUtil {


    /**
     * 兼容本地与oss存储
     * @param fileStorageDTO
     * @param textContent
     */
    public static void saveFile(FileStorageDTO fileStorageDTO,String textContent) {
        log.info("保存文件:fileStorageDTO:{}",fileStorageDTO);
        if(fileStorageDTO.isSupportOSSStorage()){
            File file = writeStringToLocalFile(System.getProperty("java.io.tmpdir"),fileStorageDTO.getSaveFileName(),textContent);
            uploadThirdStorage(fileStorageDTO, file);
            deleteFileByCondition(file,true);
        }else {
            writeStringToLocalFile(fileStorageDTO.getLocalStorageParentPath(),fileStorageDTO.getSaveFileName(),textContent);
        }
    }

    private static void uploadThirdStorage(FileStorageDTO fileStorageDTO, File file) {
        ThirdStorageFileUploadDTO thirdStorageFileUploadDTO = builderSimpleThirdStorageFileUploadDTO(
                fileStorageDTO.getOssBucketName(), fileStorageDTO.getThirdStorageParentPath(), fileStorageDTO.getSaveFileName());
        ResultDTO<ThirdStorageSimpleUploadResultDTO> thirdStorageSimpleUploadResultDTOResultDTO = ThirdStorageFileUtil.
                simpleUploadByFilePath(thirdStorageFileUploadDTO, file.getAbsolutePath());
        if (!thirdStorageSimpleUploadResultDTOResultDTO.isSuccess()) {
            deleteFileByCondition(file,true);
            throw new RuntimeException("上传到云存储失败:"+thirdStorageSimpleUploadResultDTOResultDTO.getDescription());
        }
    }

    /**
     * 兼容本地与oss存储
     * @param fileStorageDTO
     * @param bytes
     */
    public static void saveFile(FileStorageDTO fileStorageDTO,byte[] bytes) {
        log.info("保存文件:fileStorageDTO:{}",fileStorageDTO);
        if(fileStorageDTO.isSupportOSSStorage()){
            File file = writeStringToLocalFile(System.getProperty("java.io.tmpdir"),fileStorageDTO.getSaveFileName(),bytes);
            uploadThirdStorage(fileStorageDTO, file);
            deleteFileByCondition(file,true);
        }else {
            writeStringToLocalFile(fileStorageDTO.getLocalStorageParentPath(),fileStorageDTO.getSaveFileName(),bytes);
        }
    }

    /**
     * 写文件到本地存储
     * @param localStorageParentPath
     * @param saveFileName
     * @param fileItem
     * @return
     * @throws Exception
     */
    public static File writeFileItemToLocalFile(String localStorageParentPath, String saveFileName, FileItem fileItem) throws Exception {
        log.info("写文件到本地存储:storageParentPath:{},saveFileName:{}",localStorageParentPath,saveFileName);
        File parentFile = new File(localStorageParentPath);
        if (!parentFile.exists()) {
            parentFile.mkdirs();
        }
        String filePath = builderAbsoluteFilePath(localStorageParentPath, saveFileName);
        File file = new File(filePath);
        fileItem.write(file);
        return file;
    }

    private static String builderAbsoluteFilePath(String storageParentPath, String saveFileName) {
        String filePath = null;
        if(storageParentPath.endsWith("/")){
            filePath = new StringBuilder(storageParentPath).append(saveFileName).toString();
        }else {
            filePath = new StringBuilder(storageParentPath).append("/").append(saveFileName).toString();
        }
        return filePath;
    }

    /**
     * 写文件到本地存储
     * @param localStorageParentPath
     * @param saveFileName
     * @param textContent
     * @return
     */
    private static File writeStringToLocalFile(String localStorageParentPath, String saveFileName, String textContent) {
        log.info("保存文件到本地：storageParentPath:{},saveFileName:{}",localStorageParentPath,saveFileName);
        File parentFile = new File(localStorageParentPath);
        if (!parentFile.exists()) {
            parentFile.mkdirs();
        }
        String filePath = builderAbsoluteFilePath(localStorageParentPath, saveFileName);
        try(BufferedOutputStream bufferedOutputStream = new BufferedOutputStream(new FileOutputStream(filePath))) {
            byte[] bytes = textContent.getBytes("UTF-8");
            bufferedOutputStream.write(bytes);
        } catch (Exception e) {
            log.error("保存文件到本地遇到异常:",e);
            throw new RuntimeException(e);
        }
        return new File(filePath);
    }

    /**
     * 写文件到本地存储
     * @param localStorageParentPath
     * @param saveFileName
     * @param bytes
     * @return
     */
    private static File writeStringToLocalFile(String localStorageParentPath, String saveFileName, byte[] bytes) {
        log.info("保存文件到本地：storageParentPath:{},saveFileName:{}",localStorageParentPath,saveFileName);
        File parentFile = new File(localStorageParentPath);
        if (!parentFile.exists()) {
            parentFile.mkdirs();
        }
        String filePath = builderAbsoluteFilePath(localStorageParentPath, saveFileName);
        try(BufferedOutputStream bufferedOutputStream = new BufferedOutputStream(new FileOutputStream(filePath))) {
            bufferedOutputStream.write(bytes);
        } catch (Exception e) {
            log.error("保存文件到本地遇到异常:",e);
            throw new RuntimeException(e);
        }
        return new File(filePath);
    }

    public static ThirdStorageFileUploadDTO builderSimpleThirdStorageFileUploadDTO(String ossBucketName, String thirdStorageFileParentPath,
                                                                                   String saveFileName) {
        ThirdStorageFileUploadDTO thirdStorageFileUploadDTO = new ThirdStorageFileUploadDTO();
        ThirdStorageFileBaseDTO thirdStorageFileBaseDTO = new ThirdStorageFileBaseDTO();
        thirdStorageFileBaseDTO.setBucketName(ossBucketName);
        String ossFileName = builderAbsoluteFilePath(thirdStorageFileParentPath,saveFileName);
        if (ossFileName.startsWith("/")) {
            ossFileName = ossFileName.substring(1);
        }
        thirdStorageFileBaseDTO.setFileName(ossFileName);
        thirdStorageFileUploadDTO.setThirdStorageFileBaseDTO(thirdStorageFileBaseDTO);
        return thirdStorageFileUploadDTO;
    }

    /**
     * 删除单个文件
     * @param file
     * @param isDelete
     */
    public static void deleteFileByCondition(File file, boolean isDelete) {
        if (isDelete && file.exists()) {
            log.info("删除:文件路径:{}",file.getAbsoluteFile());
            file.delete();
        }
    }

    /**
     * 批量删除文件
     * @param fileDeleteDTO
     */
    public static void deleteFile(FileDeleteDTO fileDeleteDTO) {
        log.info("删除:fileDeleteDTO:{}",fileDeleteDTO);
        deleteThirdFile(fileDeleteDTO);
        deleteLocalFile(fileDeleteDTO);
    }

    private static void deleteLocalFile(FileDeleteDTO fileDeleteDTO) {
        List<String> deleteLocalFilePathList = fileDeleteDTO.getDeleteLocalFilePathList();
        for (int i = 0; i < deleteLocalFilePathList.size(); i++) {
            boolean delete = FileUtil.delete(deleteLocalFilePathList.get(i));
            if (delete) {
                log.info("删除静态文件:{}", deleteLocalFilePathList.get(i));
            } else {
                log.info("静态文件不存在，无需删除:{}", deleteLocalFilePathList.get(i));
            }
        }
    }

    private static void deleteThirdFile(FileDeleteDTO fileDeleteDTO) {
        List<String> deleteThirdFilePathList = fileDeleteDTO.getDeleteThirdFilePathList();
        if (CollectionUtils.isEmpty(deleteThirdFilePathList)) {
            return;
        }
        int pageSize = 1000;
        int totalSize = deleteThirdFilePathList.size();
        int pageNum = IntMath.divide(totalSize, pageSize, RoundingMode.CEILING);
        for (int m = 0; m < pageNum; m++) {
            ThirdStorageFileDeleteDTO thirdStorageFileDeleteDTO = new ThirdStorageFileDeleteDTO();
            thirdStorageFileDeleteDTO.setBucketName(fileDeleteDTO.getOssBucketName());
            int fromIndex = m * pageSize;
            int toIndex = pageSize;
            if (m == (pageNum-1)) {
                fromIndex = m * pageSize;
                toIndex = totalSize;
            }
            List<String> subDeleteFilePathList = deleteThirdFilePathList.subList(fromIndex, toIndex);
            thirdStorageFileDeleteDTO.setFileNameList(builderDeleteFileNameList(subDeleteFilePathList));
            ThirdStorageFileUtil.deleteFiles(thirdStorageFileDeleteDTO);
        }
    }

    private static List<String> builderDeleteFileNameList(List<String> deleteFilePathList) {
        List<String> fileNameList = Lists.newArrayList();
        for (int i = 0; i < deleteFilePathList.size(); i++) {
            String deleteFilePath = deleteFilePathList.get(i);
            if (deleteFilePath.startsWith("/")) {
                fileNameList.add(deleteFilePath.substring(1));
            } else {
                fileNameList.add(deleteFilePath);
            }
        }
        return fileNameList;
    }
}
