package com.chinamcloud.common.storage.util;

import com.chinamcloud.common.result.ResultDTO;
import com.chinamcloud.common.storage.dto.*;
import com.chinamcloud.common.storage.exception.FileStorageException;
import com.chinamcloud.common.util.FileUtil;
import com.google.common.collect.Lists;
import com.google.common.math.IntMath;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.fileupload.FileItem;

import java.io.*;
import java.math.RoundingMode;
import java.util.Collections;
import java.util.List;

/**
 * Created with IntelliJ IDEA.
 * User: chenzhiwei
 * Date: 17/2/6
 * Time: 下午2:01
 */
@Slf4j
public class FileStorageUtil {


    /**
     * 兼容本地与oss存储
     * @param fileStorageDTO
     * @param textContent
     */
    public static void saveFile(FileStorageDTO fileStorageDTO,String textContent) throws FileStorageException {
        log.info("保存文件:fileStorageDTO:{}",fileStorageDTO);
        if(fileStorageDTO.isSupportOSSStorage()){
            File file = writeStringToLocalFile(System.getProperty("java.io.tmpdir"),fileStorageDTO.getSaveFileName(),textContent);
            uploadThirdStorage(fileStorageDTO, file);
            deleteFileByCondition(file,true);
        }else {
            writeStringToLocalFile(fileStorageDTO.getLocalStorageParentPath(),fileStorageDTO.getSaveFileName(),textContent);
        }
    }

    private static void uploadThirdStorage (FileStorageDTO fileStorageDTO, File file) throws FileStorageException {
        ThirdStorageFileUploadDTO thirdStorageFileUploadDTO = builderSimpleThirdStorageFileUploadDTO(
                fileStorageDTO.getOssBucketName(), fileStorageDTO.getThirdStorageParentPath(), fileStorageDTO.getSaveFileName());
        ResultDTO<ThirdStorageSimpleUploadResultDTO> thirdStorageSimpleUploadResultDTOResultDTO = ThirdStorageFileUtil.
                simpleUploadByFilePath(thirdStorageFileUploadDTO, file.getAbsolutePath());
        if (!thirdStorageSimpleUploadResultDTOResultDTO.isSuccess()) {
            deleteFileByCondition(file,true);
            throw new FileStorageException("上传到云存储失败:"+thirdStorageSimpleUploadResultDTOResultDTO.getDescription());
        }
    }

    /**
     * 兼容本地与oss存储
     * @param fileStorageDTO
     * @param bytes
     */
    public static void saveFile(FileStorageDTO fileStorageDTO,byte[] bytes) {
        log.info("保存文件:fileStorageDTO:{}",fileStorageDTO);
        if(fileStorageDTO.isSupportOSSStorage()){
            File file = writeStringToLocalFile(System.getProperty("java.io.tmpdir"),fileStorageDTO.getSaveFileName(),bytes);
            uploadThirdStorage(fileStorageDTO, file);
            deleteFileByCondition(file,true);
        }else {
            writeStringToLocalFile(fileStorageDTO.getLocalStorageParentPath(),fileStorageDTO.getSaveFileName(),bytes);
        }
    }

    /**
     * 写文件到本地存储
     * @param localStorageParentPath
     * @param saveFileName
     * @param fileItem
     * @return
     * @throws Exception
     */
    public static File writeFileItemToLocalFile(String localStorageParentPath, String saveFileName, FileItem fileItem) throws Exception {
        log.info("写文件到本地存储:storageParentPath:{},saveFileName:{}",localStorageParentPath,saveFileName);
        File parentFile = new File(localStorageParentPath);
        if (!parentFile.exists()) {
            parentFile.mkdirs();
        }
        String filePath = PathUtil.builderPath(localStorageParentPath, saveFileName);
        File file = new File(filePath);
        fileItem.write(file);
        return file;
    }

    /**
     * 拷贝文件
     * @param sourceFilePath
     * @param targetFilePath
     * @throws IOException
     */
    public static void copyFile(String sourceFilePath,String targetFilePath) throws IOException {
        log.info("拷贝文件:sourcePath:{},targetParentPath:{}", sourceFilePath, targetFilePath);
        if (targetFilePath.lastIndexOf("/") >= 0) {
            File parentFile = new File(targetFilePath.substring(0, targetFilePath.lastIndexOf("/")));
            if (!parentFile.exists()) {
                parentFile.mkdirs();
            }
        }
        try (BufferedInputStream inputStream = new BufferedInputStream(new FileInputStream(sourceFilePath));
             BufferedOutputStream outputStream = new BufferedOutputStream(new FileOutputStream(targetFilePath))){
            int length = 0;
            byte[] buffer = new byte[1024];
            while (-1 != (length = inputStream.read(buffer))) {
                outputStream.write(buffer, 0, length);
            }
        }
    }


    /**
     * 写文件到本地存储
     * @param localStorageParentPath
     * @param saveFileName
     * @param textContent
     * @return
     */
    private static File writeStringToLocalFile(String localStorageParentPath, String saveFileName, String textContent)
            throws FileStorageException{
        log.info("保存文件到本地：storageParentPath:{},saveFileName:{}",localStorageParentPath,saveFileName);
        File parentFile = new File(localStorageParentPath);
        if (!parentFile.exists()) {
            parentFile.mkdirs();
        }
        String filePath = PathUtil.builderPath(localStorageParentPath, saveFileName);
        try(BufferedOutputStream bufferedOutputStream = new BufferedOutputStream(new FileOutputStream(filePath))) {
            byte[] bytes = textContent.getBytes("UTF-8");
            bufferedOutputStream.write(bytes);
        } catch (Exception e) {
            log.error("保存文件到本地遇到异常:",e);
            throw new FileStorageException("保存文件到本地遇到异常",e);
        }
        return new File(filePath);
    }

    /**
     * 写文件到本地存储
     * @param localStorageParentPath
     * @param saveFileName
     * @param bytes
     * @return
     */
    private static File writeStringToLocalFile(String localStorageParentPath, String saveFileName, byte[] bytes)
            throws FileStorageException{
        log.info("保存文件到本地：storageParentPath:{},saveFileName:{}",localStorageParentPath,saveFileName);
        File parentFile = new File(localStorageParentPath);
        if (!parentFile.exists()) {
            parentFile.mkdirs();
        }
        String filePath = PathUtil.builderPath(localStorageParentPath, saveFileName);
        try(BufferedOutputStream bufferedOutputStream = new BufferedOutputStream(new FileOutputStream(filePath))) {
            bufferedOutputStream.write(bytes);
        } catch (Exception e) {
            log.error("保存文件到本地遇到异常:",e);
            throw new FileStorageException("保存文件到本地遇到异常",e);
        }
        return new File(filePath);
    }

    public static ThirdStorageFileUploadDTO builderSimpleThirdStorageFileUploadDTO(String ossBucketName, String thirdStorageFileParentPath,
                                                                                   String saveFileName) {
        ThirdStorageFileUploadDTO thirdStorageFileUploadDTO = new ThirdStorageFileUploadDTO();
        ThirdStorageFileBaseDTO thirdStorageFileBaseDTO = new ThirdStorageFileBaseDTO();
        thirdStorageFileBaseDTO.setBucketName(ossBucketName);
        String ossFileName = PathUtil.builderPath(thirdStorageFileParentPath,saveFileName);
        if (ossFileName.startsWith("/")) {
            ossFileName = ossFileName.substring(1);
        }
        thirdStorageFileBaseDTO.setFileName(ossFileName);
        thirdStorageFileUploadDTO.setThirdStorageFileBaseDTO(thirdStorageFileBaseDTO);
        return thirdStorageFileUploadDTO;
    }

    /**
     * 删除单个文件
     * @param file
     * @param isDelete
     */
    public static void deleteFileByCondition(File file, boolean isDelete) {
        if (isDelete && file.exists()) {
            log.info("删除:文件路径:{}",file.getAbsoluteFile());
            file.delete();
        }
    }

    /**
     * 批量删除文件
     * @param fileDeleteDTO
     */
    public static FileDeleteResultDTO deleteFile(FileDeleteDTO fileDeleteDTO) {
        log.info("删除:fileDeleteDTO:{}",fileDeleteDTO);
        List<String> deleteThirdFileSuccessList = deleteThirdFile(fileDeleteDTO);
        List<String> deleteLocalFileSuccessList = deleteLocalFile(fileDeleteDTO.getDeleteLocalFilePathList());
        return FileDeleteResultDTO.builder().deleteThirdFileSuccessList(deleteThirdFileSuccessList).
                deleteLocalFileSuccessList(deleteLocalFileSuccessList).build();
    }

    public static List<String> deleteLocalFile(List<String> deleteLocalFilePathList) {
        if (CollectionUtils.isEmpty(deleteLocalFilePathList)){
            return Collections.emptyList();
        }
        List<String> deleteLocalFileSuccessList = Lists.newArrayList();
        for (int i = 0; i < deleteLocalFilePathList.size(); i++) {
            boolean delete = FileUtil.delete(deleteLocalFilePathList.get(i));
            if (delete) {
                deleteLocalFileSuccessList.add(deleteLocalFilePathList.get(i));
                log.info("删除静态文件:{}", deleteLocalFilePathList.get(i));
            } else {
                log.info("静态文件不存在，无需删除:{}", deleteLocalFilePathList.get(i));
            }
        }
        return deleteLocalFileSuccessList;
    }

    private static List<String> deleteThirdFile(FileDeleteDTO fileDeleteDTO) throws FileStorageException {
        List<String> deleteThirdFilePathList = fileDeleteDTO.getDeleteThirdFilePathList();
        if (CollectionUtils.isEmpty(deleteThirdFilePathList) || !fileDeleteDTO.isSupportOSSStorage()) {
            return Collections.emptyList();
        }
        int pageSize = 1000;
        int totalSize = deleteThirdFilePathList.size();
        int pageNum = IntMath.divide(totalSize, pageSize, RoundingMode.CEILING);
        List<String> successDeleteFilePath = Lists.newArrayList();
        for (int m = 0; m < pageNum; m++) {
            ThirdStorageFileDeleteDTO thirdStorageFileDeleteDTO = new ThirdStorageFileDeleteDTO();
            thirdStorageFileDeleteDTO.setBucketName(fileDeleteDTO.getOssBucketName());
            int fromIndex = m * pageSize;
            int toIndex = pageSize;
            if (m == (pageNum-1)) {
                fromIndex = m * pageSize;
                toIndex = totalSize;
            }
            List<String> subDeleteFilePathList = deleteThirdFilePathList.subList(fromIndex, toIndex);
            thirdStorageFileDeleteDTO.setFileNameList(builderDeleteFileNameList(subDeleteFilePathList));
            ResultDTO<List<String>> listResultDTO = ThirdStorageFileUtil.deleteFiles(thirdStorageFileDeleteDTO);
            if (listResultDTO.isSuccess()){
                successDeleteFilePath.addAll(listResultDTO.getData());
            }
        }
        return successDeleteFilePath;
    }

    private static List<String> builderDeleteFileNameList(List<String> deleteFilePathList) {
        List<String> fileNameList = Lists.newArrayList();
        for (int i = 0; i < deleteFilePathList.size(); i++) {
            String deleteFilePath = deleteFilePathList.get(i);
            if (deleteFilePath.startsWith("/")) {
                fileNameList.add(deleteFilePath.substring(1));
            } else {
                fileNameList.add(deleteFilePath);
            }
        }
        return fileNameList;
    }

    /**
     * 返回文件的大小
     * @param file
     * @return ：返回文件大小，单位字节
     * @throws Exception
     */
    public static long getFileSize(File file) throws Exception {
        long size = 0;
        if (!file.exists()){
            log.warn("文件路径不存在:{}",file.getAbsoluteFile());
            return size;
        }
        File files[] = file.listFiles();
        for (int i = 0; null != files && i < files.length; i++) {
            if (files[i].isDirectory()) {
                size = size + getFileSize(files[i]);
            } else {
                size = size + files[i].length();
            }
        }
        return size;
    }

    /**
     * 返回所有文件的大小
     * @param storagePathCollection
     * @return ：返回文件的大小，单位字节
     * @throws Exception
     */
    public static long getAllTotalFileSize(List<String> storagePathCollection) throws Exception {
        long totalFileSize = 0L;
        for (String storagePath : storagePathCollection) {
            totalFileSize += getFileSize(new File(storagePath));
        }
        return totalFileSize;
    }
}
