package com.chinamcloud.common.storage.util;

import com.alibaba.fastjson.JSON;
import com.aliyun.oss.ClientException;
import com.aliyun.oss.OSSClient;
import com.aliyun.oss.OSSException;
import com.aliyun.oss.model.*;
import com.chinamcloud.common.result.ResultDTO;
import com.chinamcloud.common.storage.dto.*;
import com.chinamcloud.common.storage.exception.FileStorageException;
import com.chinamcloud.common.storage.exception.ThirdStorageClientException;
import com.chinamcloud.common.storage.properties.StorageProperties;
import com.chinamcloud.common.storage.transformer.CompleteMultipartUploadResult2DTOTransformer;
import com.chinamcloud.common.storage.transformer.DownloadFileResult2DTOTransformer;
import com.chinamcloud.common.storage.transformer.ObjectMetadata2MetadataDTOTransformer;
import com.chinamcloud.common.storage.transformer.PutObjectResult2SimpleUploadResultDTOTransformer;
import com.google.common.base.MoreObjects;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;

import java.io.File;
import java.io.InputStream;
import java.util.List;
import java.util.Map;

/**
 * Created with IntelliJ IDEA.
 * User: chenzhiwei
 * Date: 17/1/9
 * Time: 下午3:02
 */
@Slf4j
public class ThirdStorageFileUtil {

    private static final String DELIMITER="/";
    /**
     * 简单的上传文件：通过输入流的方式
     *
     * @param thirdStorageFileUploadDTO
     * @param inputStream
     * @return
     * @throws FileStorageException
     */
    public static ResultDTO<ThirdStorageSimpleUploadResultDTO> simpleUploadByInputStream(
            ThirdStorageFileUploadDTO thirdStorageFileUploadDTO, InputStream inputStream)
            throws FileStorageException ,ThirdStorageClientException{
        ResultDTO<Boolean> booleanResultDTO = checkFileUploadDTO(thirdStorageFileUploadDTO);
        if (!booleanResultDTO.isSuccess()) {
            return ResultDTO.fail(booleanResultDTO.getDescription());
        }
        if (inputStream == null) {
            return ResultDTO.fail("input inputStream is not null");
        }
        ThirdStorageFileBaseDTO fileBaseDTO = thirdStorageFileUploadDTO.getThirdStorageFileBaseDTO();
        OSSClient ossClient = OSSClientUtil.createOssClient(fileBaseDTO.getEndpoint());
        ObjectMetadata objectMetadata = getObjectMetadata(thirdStorageFileUploadDTO);
        PutObjectRequest putObjectRequest = new PutObjectRequest(fileBaseDTO.getBucketName(), fileBaseDTO.getFileName(),
                inputStream, objectMetadata);
        try {
            PutObjectResult putObjectResult = ossClient.putObject(putObjectRequest);
            log.info("putObjectResult:{}", JSON.toJSONString(putObjectResult, true));
            ThirdStorageSimpleUploadResultDTO thirdStorageSimpleUploadResultDTO = builderSimpleUploadResultDTO(fileBaseDTO, putObjectResult);
            log.info("thirdStorageSimpleUploadResultDTO:{}", JSON.toJSONString(thirdStorageSimpleUploadResultDTO, true));
            return ResultDTO.successfy(thirdStorageSimpleUploadResultDTO);
        } catch (OSSException ossException) {
            log.error("上传文件遇到OSSException异常", ossException);
            throw new FileStorageException("上传oss遇到异常");
        } catch (ClientException e) {
            log.error("上传文件遇到ClientException异常", e);
            throw new ThirdStorageClientException("上传oss遇到异常");
        } finally {
            ossClient.shutdown();
        }
    }

    private static ThirdStorageSimpleUploadResultDTO builderSimpleUploadResultDTO(ThirdStorageFileBaseDTO thirdStorageFileBaseDTO,
                                                                                  PutObjectResult putObjectResult) {
        ThirdStorageSimpleUploadResultDTO thirdStorageSimpleUploadResultDTO = PutObjectResult2SimpleUploadResultDTOTransformer.
                INSTANCE.apply(putObjectResult);
        String location = builderLocation(thirdStorageFileBaseDTO);
        thirdStorageSimpleUploadResultDTO.setLocation(location);
        return thirdStorageSimpleUploadResultDTO;
    }

    private static String builderLocation(ThirdStorageFileBaseDTO thirdStorageFileBaseDTO) {
        StringBuilder stringBuilder = new StringBuilder();
        String endPoint = MoreObjects.firstNonNull(thirdStorageFileBaseDTO.getEndpoint(), StorageProperties.getInstance().
                getOss_default_endpoint());
        endPoint = endPoint.replace("://", "://" + thirdStorageFileBaseDTO.getBucketName() + ".");
        stringBuilder.append(endPoint);
        stringBuilder.append("/");
        stringBuilder.append(thirdStorageFileBaseDTO.getFileName());
        return stringBuilder.toString();
    }

    /**
     * 简单的上传文件
     *
     * @param thirdStorageFileUploadDTO
     * @param uploadFilePath
     * @return
     * @throws FileStorageException
     */
    public static ResultDTO<ThirdStorageSimpleUploadResultDTO> simpleUploadByFilePath(
            ThirdStorageFileUploadDTO thirdStorageFileUploadDTO, String uploadFilePath) throws FileStorageException,
            ThirdStorageClientException {
        ResultDTO<Boolean> booleanResultDTO = checkFileUploadDTO(thirdStorageFileUploadDTO);
        if (!booleanResultDTO.isSuccess()) {
            return ResultDTO.fail(booleanResultDTO.getDescription());
        }
        if (StringUtils.isBlank(uploadFilePath)) {
            return ResultDTO.fail("input filePath is not null");
        }
        File file = new File(uploadFilePath);
        if(!file.exists()){
            log.error("上传oss的文件不存在:{},{}",thirdStorageFileUploadDTO,uploadFilePath);
            return ResultDTO.fail("上传oss的文件不存在");
        }
        ThirdStorageFileBaseDTO fileBaseDTO = thirdStorageFileUploadDTO.getThirdStorageFileBaseDTO();
        OSSClient ossClient = OSSClientUtil.createOssClient(fileBaseDTO.getEndpoint());
        ObjectMetadata objectMetadata = getObjectMetadata(thirdStorageFileUploadDTO);
        PutObjectRequest putObjectRequest = new PutObjectRequest(fileBaseDTO.getBucketName(), fileBaseDTO.getFileName(),
                file, objectMetadata);
        try {
            PutObjectResult putObjectResult = ossClient.putObject(putObjectRequest);
            log.info("putObjectResult:{}", JSON.toJSONString(putObjectResult, true));
            ThirdStorageSimpleUploadResultDTO thirdStorageSimpleUploadResultDTO = builderSimpleUploadResultDTO(fileBaseDTO, putObjectResult);
            log.info("thirdStorageSimpleUploadResultDTO:{}", JSON.toJSONString(thirdStorageSimpleUploadResultDTO, true));
            return ResultDTO.successfy(thirdStorageSimpleUploadResultDTO);
        } catch (OSSException ossException) {
            log.error("上传文件遇到OSSException异常", ossException);
            throw new FileStorageException("上传oss遇到异常");
        } catch (ClientException e) {
            log.error("上传文件遇到ClientException异常", e);
            throw new ThirdStorageClientException("上传oss遇到异常");
        } finally {
            ossClient.shutdown();
        }
    }

    /**
     * 递归文件上传
     * @param thirdStorageRecursionFileUploadDTO
     * @return
     * @throws FileStorageException
     */
    public static ResultDTO<List<ThirdStorageSimpleUploadResultDTO>> recursionSimpleUploadByFilePath(
            ThirdStorageRecursionFileUploadDTO thirdStorageRecursionFileUploadDTO) throws FileStorageException {
        log.info("thirdStorageRecursionFileUploadDTO:{}", thirdStorageRecursionFileUploadDTO);
        ResultDTO<Boolean> booleanResultDTO = checkThirdStorageRecursiveFileUploadDTO(thirdStorageRecursionFileUploadDTO);
        if (!booleanResultDTO.isSuccess()) {
            return ResultDTO.fail(booleanResultDTO.getDescription());
        }
        List<ThirdStorageSimpleUploadResultDTO> thirdStorageSimpleUploadResultDTOList = Lists.newArrayList();
        for (ThirdStorageUploadFilePathDTO thirdStorageUploadFilePathDTO : thirdStorageRecursionFileUploadDTO.getThirdStorageUploadFilePathDTOList()){
            File file = new File(thirdStorageUploadFilePathDTO.getUploadFilePath());
            recursionUploadFile(thirdStorageRecursionFileUploadDTO, thirdStorageSimpleUploadResultDTOList, thirdStorageUploadFilePathDTO,file);
        }
        return ResultDTO.successfy(thirdStorageSimpleUploadResultDTOList);
    }

    private static void recursionUploadFile(ThirdStorageRecursionFileUploadDTO thirdStorageRecursionFileUploadDTO,
                                            List<ThirdStorageSimpleUploadResultDTO> thirdStorageSimpleUploadResultDTOList,
                                            ThirdStorageUploadFilePathDTO thirdStorageUploadFilePathDTO,File parentFile) {
        if (parentFile.isDirectory()) {
            File[] files = parentFile.listFiles();
            for (File file : files) {
                if (file.isDirectory()) {
                    recursionUploadFile(thirdStorageRecursionFileUploadDTO, thirdStorageSimpleUploadResultDTOList,
                            thirdStorageUploadFilePathDTO,file);
                } else {
                    uploadFile(thirdStorageRecursionFileUploadDTO, thirdStorageSimpleUploadResultDTOList,thirdStorageUploadFilePathDTO,file);
                }
            }
        } else {
            uploadFile(thirdStorageRecursionFileUploadDTO, thirdStorageSimpleUploadResultDTOList,thirdStorageUploadFilePathDTO, parentFile);
        }
    }

    private static void uploadFile(ThirdStorageRecursionFileUploadDTO thirdStorageRecursionFileUploadDTO,
                                   List<ThirdStorageSimpleUploadResultDTO> thirdStorageSimpleUploadResultDTOList,
                                   ThirdStorageUploadFilePathDTO thirdStorageUploadFilePathDTO, File file) {
        ThirdStorageFileUploadDTO thirdStorageFileUploadDTO = new ThirdStorageFileUploadDTO();
        ThirdStorageFileBaseDTO thirdStorageFileBaseDTO = new ThirdStorageFileBaseDTO();
        thirdStorageFileBaseDTO.setEndpoint(thirdStorageRecursionFileUploadDTO.getEndpoint());
        thirdStorageFileBaseDTO.setBucketName(thirdStorageRecursionFileUploadDTO.getBucketName());
        String ossFileName = getOssFileName(thirdStorageUploadFilePathDTO, file);
        thirdStorageFileBaseDTO.setFileName(ossFileName);
        thirdStorageFileUploadDTO.setThirdStorageFileBaseDTO(thirdStorageFileBaseDTO);
        ResultDTO<ThirdStorageSimpleUploadResultDTO> thirdStorageSimpleUploadResultDTOResultDTO = simpleUploadByFilePath(
                thirdStorageFileUploadDTO, file.getAbsolutePath());
        if (thirdStorageSimpleUploadResultDTOResultDTO.isSuccess()) {
            thirdStorageSimpleUploadResultDTOList.add(thirdStorageSimpleUploadResultDTOResultDTO.getData());
        }
    }

    private static String getOssFileName(ThirdStorageUploadFilePathDTO thirdStorageUploadFilePathDTO, File file) {
        String ossFileName = file.getAbsolutePath().replace(thirdStorageUploadFilePathDTO.getUploadFilePath(),"");
        String reservedUploadPath = thirdStorageUploadFilePathDTO.getReservedUploadPath();
        if(StringUtils.isNotBlank(reservedUploadPath)){
            ossFileName = PathUtil.builderPath(reservedUploadPath,ossFileName);
        }
        if (ossFileName.startsWith("/")) {
            ossFileName = ossFileName.substring(1);
        }
        return ossFileName;
    }


    private static ResultDTO<Boolean> checkThirdStorageRecursiveFileUploadDTO(ThirdStorageRecursionFileUploadDTO thirdStorageRecursionFileUploadDTO) {
        if (thirdStorageRecursionFileUploadDTO == null) {
            return ResultDTO.fail("thirdStorageRecursionFileUploadDTO is not null");
        }
        if (StringUtils.isBlank(thirdStorageRecursionFileUploadDTO.getBucketName())) {
            return ResultDTO.fail("bucketName is not null");
        }
        if (CollectionUtils.isEmpty(thirdStorageRecursionFileUploadDTO.getThirdStorageUploadFilePathDTOList())) {
            return ResultDTO.fail("thirdStorageUploadFilePathDTOList is not null");
        }
        return ResultDTO.successfy(true);
    }

    /**
     * 断点续传：上传文件
     *
     * @param thirdStorageFileUploadDTO：文件的基本信息
     * @param uploadFilePath：上传文件的地址
     * @return
     * @throws FileStorageException
     */
    public static ResultDTO<CompleteMultipartUploadResultDTO> uploadByBreakpointsContinuedTransmission(
            ThirdStorageFileUploadDTO thirdStorageFileUploadDTO, String uploadFilePath) throws FileStorageException {
        ResultDTO<Boolean> booleanResultDTO = checkFileUploadDTO(thirdStorageFileUploadDTO);
        if (!booleanResultDTO.isSuccess()) {
            return ResultDTO.fail(booleanResultDTO.getDescription());
        }
        if (StringUtils.isBlank(uploadFilePath)) {
            return ResultDTO.fail("uploadFilePath is not null");
        }
        ThirdStorageFileBaseDTO fileBaseDTO = thirdStorageFileUploadDTO.getThirdStorageFileBaseDTO();
        OSSClient ossClient = OSSClientUtil.createOssClient(fileBaseDTO.getEndpoint());
        UploadFileRequest uploadFileRequest = new UploadFileRequest(fileBaseDTO.getBucketName(), fileBaseDTO.getFileName());
        uploadFileRequest.setUploadFile(uploadFilePath);//待上传的本地文件
        uploadFileRequest.setTaskNum(StorageProperties.getInstance().getOss_taskNum());//设置并发上传数
        uploadFileRequest.setPartSize(StorageProperties.getInstance().getOss_partSize());//设置分片大小
        uploadFileRequest.setEnableCheckpoint(true);//开启断点续传，默认关闭
        ObjectMetadata objectMetadata = getObjectMetadata(thirdStorageFileUploadDTO);
        uploadFileRequest.setObjectMetadata(objectMetadata);
        try {
            UploadFileResult uploadFileResult = ossClient.uploadFile(uploadFileRequest);
            CompleteMultipartUploadResult completeMultipartUploadResult = uploadFileResult.getMultipartUploadResult();
            log.info("completeMultipartUploadResult：{}", JSON.toJSONString(completeMultipartUploadResult, true));
            CompleteMultipartUploadResultDTO completeMultipartUploadResultDTO = CompleteMultipartUploadResult2DTOTransformer.
                    INSTANCE.apply(completeMultipartUploadResult);
            log.info("completeMultipartUploadResultDTO:{}", JSON.toJSONString(completeMultipartUploadResultDTO, true));
            return ResultDTO.successfy(completeMultipartUploadResultDTO);
        } catch (Throwable throwable) {
            log.error("断点续传-上传-遇到异常", throwable);
            throw new FileStorageException("上传oss遇到异常");
        } finally {
            ossClient.shutdown();
        }
    }

    private static ObjectMetadata getObjectMetadata(ThirdStorageFileUploadDTO thirdStorageFileUploadDTO) {
        if (thirdStorageFileUploadDTO.getMetadataDTO() == null) {
            return null;
        }
        Map<String, Object> metadataMap = thirdStorageFileUploadDTO.getMetadataDTO().getMetadataMap();
        ObjectMetadata objectMetadata = new ObjectMetadata();
        if (MapUtils.isNotEmpty(metadataMap)) {
            for (String key : metadataMap.keySet()) {
                objectMetadata.setHeader(key, metadataMap.get(key));
            }
        }
        Map<String, String> userMetadataMap = thirdStorageFileUploadDTO.getMetadataDTO().getUserMetadataMap();
        if (MapUtils.isNotEmpty(userMetadataMap)) {
            objectMetadata.setUserMetadata(userMetadataMap);
        }
        return objectMetadata;
    }


    private static ResultDTO<Boolean> checkFileUploadDTO(ThirdStorageFileUploadDTO thirdStorageFileUploadDTO) {
        if (thirdStorageFileUploadDTO == null) {
            return ResultDTO.fail("fileUploadBaseInfoDTO is not null");
        }
        ResultDTO<Boolean> booleanResultDTO = checkFileBaseDTO(thirdStorageFileUploadDTO.getThirdStorageFileBaseDTO());
        if (!booleanResultDTO.isSuccess()) {
            return booleanResultDTO;
        }
        return ResultDTO.successfy(true);
    }

    private static ResultDTO<Boolean> checkFileBaseDTO(ThirdStorageFileBaseDTO thirdStorageFileBaseDTO) {
        if (thirdStorageFileBaseDTO == null) {
            return ResultDTO.fail("thirdStorageFileBaseDTO is not null");
        }
        if (StringUtils.isBlank(thirdStorageFileBaseDTO.getBucketName())) {
            return ResultDTO.fail("bucketName is not null");
        }
        if (StringUtils.isBlank(thirdStorageFileBaseDTO.getFileName())) {
            return ResultDTO.fail("fileName is not null");
        }
        String fileName = thirdStorageFileBaseDTO.getFileName();
        String firstString = fileName.substring(0, 1);
        if ("/".equals(firstString) || "\\".equals(firstString)) {
            return ResultDTO.fail("不能以'/'或者'\'字符开头");
        }
        if (fileName.length() >= 1023) {
            return ResultDTO.fail("1 <= fileName.length < 1023");
        }
        return ResultDTO.successfy(true);
    }

    /**
     * 简单的文件下载
     *
     * @param thirdStorageFileBaseDTO
     * @param downloadSaveFilePath
     * @return
     * @throws FileStorageException
     */
    public static ResultDTO<MetadataDTO> simpleDownloadToFile(ThirdStorageFileBaseDTO thirdStorageFileBaseDTO, String downloadSaveFilePath)
            throws FileStorageException,ThirdStorageClientException {
        ResultDTO<Boolean> booleanResultDTO = checkFileBaseDTO(thirdStorageFileBaseDTO);
        if (!booleanResultDTO.isSuccess()) {
            return ResultDTO.fail(booleanResultDTO.getDescription());
        }
        if (StringUtils.isBlank(downloadSaveFilePath)) {
            return ResultDTO.fail("input filePath is null");
        }
        OSSClient ossClient = OSSClientUtil.createOssClient(thirdStorageFileBaseDTO.getEndpoint());
        try {
            ObjectMetadata objectMetadata = ossClient.getObject(new GetObjectRequest(thirdStorageFileBaseDTO.getBucketName(),
                            thirdStorageFileBaseDTO.getFileName()),
                    new File(downloadSaveFilePath));
            MetadataDTO metadataDTO = ObjectMetadata2MetadataDTOTransformer.INSTANCE.apply(objectMetadata);
            log.info("metadataDTO:{}", JSON.toJSONString(metadataDTO, true));
            return ResultDTO.successfy(metadataDTO);
        } catch (OSSException ossException) {
            log.error("下载文件遇到OSSException异常", ossException);
            throw new FileStorageException("oss下载遇到异常");
        } catch (ClientException e) {
            log.error("下载文件遇到ClientException异常", e);
            throw new ThirdStorageClientException("oss下载遇到异常");
        } finally {
            ossClient.shutdown();
        }
    }

    /**
     * 断点续传：下载文件
     *
     * @param thirdStorageFileBaseDTO：文件的基本信息
     * @param downloadSaveFilePath：文件下载后保存的路径
     * @return
     * @throws FileStorageException
     */
    public static ResultDTO<ThirdStorageDownloadFileResultDTO> downloadByBreakpointsContinuedTransmission(
            ThirdStorageFileBaseDTO thirdStorageFileBaseDTO, String downloadSaveFilePath) throws FileStorageException {
        ResultDTO<Boolean> booleanResultDTO = checkFileBaseDTO(thirdStorageFileBaseDTO);
        if (!booleanResultDTO.isSuccess()) {
            return ResultDTO.fail(booleanResultDTO.getDescription());
        }
        if (StringUtils.isBlank(downloadSaveFilePath)) {
            return ResultDTO.fail("downloadFilePath is not null");
        }
        OSSClient ossClient = OSSClientUtil.createOssClient(thirdStorageFileBaseDTO.getEndpoint());
        DownloadFileRequest downloadFileRequest = new DownloadFileRequest(thirdStorageFileBaseDTO.getBucketName(),
                thirdStorageFileBaseDTO.getFileName());
        downloadFileRequest.setDownloadFile(downloadSaveFilePath);//设置下载后保存到本地文件的路径
        downloadFileRequest.setTaskNum(StorageProperties.getInstance().getOss_taskNum());//设置并发下载数
        downloadFileRequest.setPartSize(StorageProperties.getInstance().getOss_partSize());//设置分片大小
        downloadFileRequest.setEnableCheckpoint(true);//开启断点续传，默认关闭
        try {
            ThirdStorageDownloadFileResultDTO thirdStorageDownloadFileResultDTO = DownloadFileResult2DTOTransformer.INSTANCE.
                    apply(ossClient.downloadFile(downloadFileRequest));
            log.info("thirdStorageDownloadFileResultDTO:{}", JSON.toJSONString(thirdStorageDownloadFileResultDTO, true));
            return ResultDTO.successfy(thirdStorageDownloadFileResultDTO);
        } catch (Throwable throwable) {
            log.error("断点续传-下载-遇到异常", throwable);
            throw new FileStorageException("oss下载遇到异常");
        } finally {
            ossClient.shutdown();
        }
    }

    /**
     * 文件删除：调用方根据返回成功的数量与删除的数量进行对比以确认是否全部删除成功
     * 单次批量删除的上限是1000条
     * 即使阿里云环境没有该文件，也算删除成功
     *
     * @param thirdStorageFileDeleteDTO
     * @return ：返回删除成功的文件
     * @throws FileStorageException
     */
    public static ResultDTO<List<String>> deleteFiles(ThirdStorageFileDeleteDTO thirdStorageFileDeleteDTO) throws
            FileStorageException,ThirdStorageClientException {
        ResultDTO<Boolean> booleanResultDTO = checkFileDeleteDTO(thirdStorageFileDeleteDTO);
        if (!booleanResultDTO.isSuccess()) {
            return ResultDTO.fail(booleanResultDTO.getDescription());
        }
        OSSClient ossClient = OSSClientUtil.createOssClient(thirdStorageFileDeleteDTO.getEndpoint());
        //quiet用于设定返回模式
        //详细模式：false,返回的成功删除Object的结果---默认模式
        //简单模式：true,返回删除过程中出错的Object结果
        DeleteObjectsRequest deleteObjectsRequest = new DeleteObjectsRequest(thirdStorageFileDeleteDTO.getBucketName()).
                withKeys(thirdStorageFileDeleteDTO.getFileNameList());
        deleteObjectsRequest.setQuiet(false);//返回成功删除Object的结果
        try {
            log.info("待删除的云存储文件:{}", deleteObjectsRequest.getKeys());
            DeleteObjectsResult deleteObjectsResult = ossClient.deleteObjects(deleteObjectsRequest);
            List<String> deletedObjects = deleteObjectsResult.getDeletedObjects();
            log.info("成功删除云存储文件:{}", deletedObjects);
            if (CollectionUtils.isEmpty(deletedObjects)) {
                return ResultDTO.fail("全部文件删除失败");
            }
            return ResultDTO.successfy(deletedObjects);
        } catch (OSSException ossException) {
            log.error("删除文件遇到OSSException异常", ossException);
            throw new FileStorageException("oss删除文件遇到异常");
        } catch (ClientException e) {
            log.error("删除文件遇到ClientException异常", e);
            throw new ThirdStorageClientException("oss删除文件遇到异常");
        } finally {
            ossClient.shutdown();
        }
    }

    private static ResultDTO<Boolean> checkFileDeleteDTO(ThirdStorageFileDeleteDTO thirdStorageFileDeleteDTO) {
        if (thirdStorageFileDeleteDTO == null) {
            return ResultDTO.fail("fileDeleteDTO is not null");
        }
        if (StringUtils.isBlank(thirdStorageFileDeleteDTO.getBucketName())) {
            return ResultDTO.fail("bucketName is not null");
        }
        if (CollectionUtils.isEmpty(thirdStorageFileDeleteDTO.getFileNameList())) {
            return ResultDTO.fail("fileNameList is not empty");
        }
        if (thirdStorageFileDeleteDTO.getFileNameList().size() > 1000) {
            return ResultDTO.fail("fileNameLise size <=1000 ");
        }
        return ResultDTO.successfy(true);
    }

    /**
     * 确定文件是否存在
     *
     * @param thirdStorageFileBaseDTO
     * @return true:存在，false:不存在
     * @throws FileStorageException
     */
    public static ResultDTO<Boolean> isExistFile(ThirdStorageFileBaseDTO thirdStorageFileBaseDTO) throws FileStorageException,
            ThirdStorageClientException {
        ResultDTO<Boolean> booleanResultDTO = checkFileBaseDTO(thirdStorageFileBaseDTO);
        if (!booleanResultDTO.isSuccess()) {
            return booleanResultDTO;
        }
        OSSClient ossClient = OSSClientUtil.createOssClient(thirdStorageFileBaseDTO.getEndpoint());
        try {
            boolean isExist = ossClient.doesObjectExist(thirdStorageFileBaseDTO.getBucketName(), thirdStorageFileBaseDTO.getFileName());
            return ResultDTO.successfy(isExist);
        } catch (OSSException ossException) {
            log.error("确定文件是否存在遇到OSSException异常", ossException);
            throw new FileStorageException("oss确定文件是否存在遇到异常");
        } catch (ClientException e) {
            log.error("确定文件是否存在遇到ClientException异常", e);
            throw new ThirdStorageClientException("oss确定文件是否存在遇到异常");
        } finally {
            ossClient.shutdown();
        }
    }

    /**
     * 查询目录，返回文件和子目录
     * @param thirdStorageFileBaseDTO
     * @return
     * @throws FileStorageException
     * @throws ThirdStorageClientException
     */
    public static ResultDTO<ThirdDirectoryDTO> queryDirectory(ThirdStorageFileBaseDTO thirdStorageFileBaseDTO)
            throws FileStorageException,ThirdStorageClientException{
        ResultDTO<Boolean> booleanResultDTO = checkFileBaseDTO(thirdStorageFileBaseDTO);
        if (!booleanResultDTO.isSuccess()) {
            return ResultDTO.fail(booleanResultDTO.getDescription());
        }
        OSSClient ossClient = OSSClientUtil.createOssClient(thirdStorageFileBaseDTO.getEndpoint());
        ThirdDirectoryDTO thirdDirectoryDTO = new ThirdDirectoryDTO();
        List<String> filePathList = Lists.newArrayList();
        List<String> directoryList = Lists.newArrayList();
        try {
            ListObjectsRequest listObjectsRequest = new ListObjectsRequest(thirdStorageFileBaseDTO.getBucketName());
            listObjectsRequest.setDelimiter(DELIMITER);
            listObjectsRequest.setPrefix(PathUtil.builderPathEndSlash(thirdStorageFileBaseDTO.getFileName()));
            ObjectListing objectListing = ossClient.listObjects(listObjectsRequest);
            for (OSSObjectSummary ossObjectSummary : objectListing.getObjectSummaries()){
                filePathList.add(ossObjectSummary.getKey());
            }
            for (String commonPrefix : objectListing.getCommonPrefixes()){
                directoryList.add(commonPrefix);
            }
        } catch (OSSException ossException) {
            log.error("查询目录遇到OSSException异常", ossException);
            throw new FileStorageException("oss确定文件是否存在遇到异常");
        } catch (ClientException e) {
            log.error("查询目录遇到ClientException异常", e);
            throw new ThirdStorageClientException("oss确定文件是否存在遇到异常");
        } finally {
            ossClient.shutdown();
        }
        thirdDirectoryDTO.setFilePathList(filePathList);
        thirdDirectoryDTO.setDirectoryList(directoryList);
        return ResultDTO.successfy(thirdDirectoryDTO);
    }

}
