package com.chinamcloud.spider.utils;

import com.alibaba.fastjson.JSONObject;
import com.chinamcloud.spider.dto.InterfaceDto;
import com.chinamcloud.spider.dto.MultipartBodyPartDto;
import com.google.common.base.MoreObjects;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import okhttp3.Headers;
import okhttp3.Response;
import org.apache.commons.collections.MapUtils;
import org.springframework.util.MultiValueMap;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.multipart.MultipartHttpServletRequest;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.URLDecoder;
import java.util.*;

/**
 * Created with IntelliJ IDEA.
 * User: chenzhiwei
 * Date: 17/9/15
 * Time: 上午11:07
 */
@Slf4j
public class RequestUtil {
    public static String getValue(HttpServletRequest request, String key) {
        return request.getHeader(key) != null ? request.getHeader(key) : request.getParameter(key);
    }

    public static List<MultipartBodyPartDto> getMultipartBodyPartDtoList(HttpServletRequest request) throws IOException {
        MultiValueMap<String, MultipartFile> multiFileMap = ((MultipartHttpServletRequest) request).getMultiFileMap();
        if (MapUtils.isEmpty(multiFileMap)) {
            return Lists.newArrayList();
        }
        List<MultipartBodyPartDto> multipartBodyPartDtoList = Lists.newArrayList();
        for (Map.Entry<String, List<MultipartFile>> entry : multiFileMap.entrySet()) {
            List<MultipartFile> multipartFileList = entry.getValue();
            for (MultipartFile multipartFile : multipartFileList) {
                MultipartBodyPartDto multipartBodyPartDto = new MultipartBodyPartDto();
                multipartBodyPartDto.setName(multipartFile.getName());
                multipartBodyPartDto.setOriginalFilename(multipartFile.getOriginalFilename());
                multipartBodyPartDto.setInputStream(multipartFile.getInputStream());
                multipartBodyPartDtoList.add(multipartBodyPartDto);
            }
        }
        return multipartBodyPartDtoList;
    }

    public static void forward(HttpServletRequest request, HttpServletResponse response, String domainUrl, String servletPath, RequestMethod requestMethod) {
        try {
            InterfaceDto interfaceDto = new InterfaceDto();
            interfaceDto.setUrl(PathUtil.builderPath(domainUrl, MoreObjects.firstNonNull(servletPath, request.getServletPath())));
            Map<String, String> headerMap = buildRequestHeaders(request);
            if (!StringUtils.isEmpty(request.getContentType())) {
                headerMap.put("contentType", request.getContentType());
            }
            if (requestMethod != null) {
                headerMap.put("method",requestMethod.name());
            }
            interfaceDto.setHeaderMap(headerMap);
            Map<String, Object> paramMap = buildRequestQueryParams(request);
            interfaceDto.setParameterMap(paramMap);
            if (request instanceof MultipartHttpServletRequest) {
                List<MultipartBodyPartDto> multipartBodyPartDtoList = getMultipartBodyPartDtoList(request);
                interfaceDto.setMultipartBodyPartDtoList(multipartBodyPartDtoList);
            }
            Response result = OkHttpUtil2.execute(interfaceDto);
            Headers headers = result.headers();
            if (headers != null) {
                Set<String> names = headers.names();
                for (String name : names) {
                    response.addHeader(name, headers.get(name));
                }
            }
            response.setStatus(result.code());
            response.getWriter().write(result.body().string());
        } catch (Exception e) {
            log.error("转发遇到异常", e);
            throw new RuntimeException(e);
        }
    }


    private static Map<String, String> buildRequestHeaders(HttpServletRequest request) {
        Map<String, String> headerMap = Maps.newHashMap();
        Enumeration<String> enumeration = request.getHeaderNames();
        if (enumeration != null) {
            while (enumeration.hasMoreElements()) {
                String name = enumeration.nextElement();
                if ("content-length".equalsIgnoreCase(name)) {
                    continue;
                }
                headerMap.put(name, request.getHeader(name));
            }
        }
        return headerMap;
    }

    private static Map<String, Object> buildRequestQueryParams(HttpServletRequest request) {
        Map<String, Object> queryParamsMap = getQueryParams(request);
        Enumeration<String> enumeration = request.getParameterNames();
        if (enumeration != null) {
            while (enumeration.hasMoreElements()) {
                String name = enumeration.nextElement();
                queryParamsMap.put(name, request.getParameter(name));
            }
        }
        try {
            BufferedReader bufferedReader = new BufferedReader(new InputStreamReader(request.getInputStream()));
            StringBuilder stringBuilder = new StringBuilder();
            String inputStr = null;
            while ((inputStr = bufferedReader.readLine()) != null) {
                stringBuilder.append(inputStr);
            }
            String jsonString = stringBuilder.toString();
            if (!StringUtils.isEmpty(jsonString)) {
                queryParamsMap.putAll(JSONObject.parseObject(jsonString));
            }

        } catch (IOException e) {
            e.printStackTrace();
        }
        return queryParamsMap;
    }

    private static Map<String, Object> getQueryParams(HttpServletRequest request) {
        Map<String, Object> qp = Maps.newHashMap();
        if (request.getQueryString() == null) {
            return qp;
        }
        StringTokenizer st = new StringTokenizer(request.getQueryString(), "&");
        int i;
        while (st.hasMoreTokens()) {
            String s = st.nextToken();
            i = s.indexOf("=");
            if (i > 0 && s.length() >= i + 1) {
                String name = s.substring(0, i);
                String value = s.substring(i + 1);
                try {
                    name = URLDecoder.decode(name, "UTF-8");
                } catch (Exception e) {
                }
                try {
                    value = URLDecoder.decode(value, "UTF-8");
                } catch (Exception e) {
                }
                qp.put(name, value);
            } else if (i == -1) {
                String name = s;
                String value = "";
                try {
                    name = URLDecoder.decode(name, "UTF-8");
                } catch (Exception e) {
                }
                qp.put(name, value);
            }
        }
        return qp;
    }
}
