package com.chinamcloud.spider.utils;

import com.chinamcloud.spider.base.ResultDTO;
import com.google.common.base.MoreObjects;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.ConversionNotSupportedException;
import org.springframework.beans.TypeMismatchException;
import org.springframework.http.converter.HttpMessageNotReadableException;
import org.springframework.http.converter.HttpMessageNotWritableException;
import org.springframework.web.HttpMediaTypeNotAcceptableException;
import org.springframework.web.HttpRequestMethodNotSupportedException;
import org.springframework.web.bind.MissingServletRequestParameterException;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ResponseBody;

import java.io.IOException;

/**
 * 异常增强 以JSON的形式返回给客服端
 * Created by jyy on 17/6/8.
 * <p>
 * 异常增强类型：NullPointerException,RunTimeException,ClassCastException,NoSuchMethodException,IOException,IndexOutOfBoundsException
 * Exception                       HTTP Status Code
 * ConversionNotSupportedException         500 (Internal Server Error)
 * HttpMessageNotWritableException         500 (Internal Server Error)
 * HttpMediaTypeNotSupportedException      415 (Unsupported Media Type)
 * HttpMediaTypeNotAcceptableException     406 (Not Acceptable)
 * HttpRequestMethodNotSupportedException  405 (Method Not Allowed)
 * NoSuchRequestHandlingMethodException    404 (Not Found)
 * TypeMismatchException                   400 (Bad Request)
 * HttpMessageNotReadableException         400 (Bad Request)
 * MissingServletRequestParameterException 400 (Bad Request)
 */
@ControllerAdvice
@Slf4j
public class SpiderExceptionHandler {

    //运行时异常
    @ExceptionHandler(RuntimeException.class)
    @ResponseBody
    public ResultDTO runtimeExceptionHandler(RuntimeException ex) {
        log.error("全局异常处理 错误原因(运行时异常) {}", ex);
        return ResultDTO.fail(MoreObjects.firstNonNull(ex.getMessage(), "操作失败，服务遇到异常"));
    }

    //空指针异常
    @ExceptionHandler(NullPointerException.class)
    @ResponseBody
    public ResultDTO nullPointerExceptionHandler(NullPointerException ex) {
        log.error("全局异常处理 错误原因(空指针异常) {}", ex);
        return ResultDTO.fail();
    }

    //类型转换异常
    @ExceptionHandler(ClassCastException.class)
    @ResponseBody
    public ResultDTO classCastExceptionHandler(ClassCastException ex) {
        log.error("全局异常处理 错误原因(类型转换异常) {}", ex);
        return ResultDTO.fail();
    }

    //IO异常
    @ExceptionHandler(IOException.class)
    @ResponseBody
    public ResultDTO iOExceptionHandler(IOException ex) {
        log.error("全局异常处理 错误原因(IO异常) {}", ex);
        return ResultDTO.fail();
    }

    //未知方法异常
    @ExceptionHandler(NoSuchMethodException.class)
    @ResponseBody
    public ResultDTO noSuchMethodExceptionHandler(NoSuchMethodException ex) {
        log.error("全局异常处理 错误原因(未知方法异常) {}", ex);
        return ResultDTO.fail();
    }

    //数组越界异常
    @ExceptionHandler(IndexOutOfBoundsException.class)
    @ResponseBody
    public ResultDTO indexOutOfBoundsExceptionHandler(IndexOutOfBoundsException ex) {
        log.error("全局异常处理 错误原因(数组越界异常) {}", ex);
        return ResultDTO.fail();
    }

    //400错误
    @ExceptionHandler({HttpMessageNotReadableException.class})
    @ResponseBody
    public ResultDTO requestNotReadable(HttpMessageNotReadableException ex) {
        log.error("全局异常处理 错误原因(400错误) {}", ex);
        return ResultDTO.fail();
    }

    //400错误
    @ExceptionHandler({TypeMismatchException.class})
    @ResponseBody
    public ResultDTO requestTypeMismatch(TypeMismatchException ex) {
        log.error("全局异常处理 错误原因(类型不支持) {}", ex);
        return ResultDTO.fail();
    }

    //400错误
    @ExceptionHandler({MissingServletRequestParameterException.class})
    @ResponseBody
    public ResultDTO requestMissingServletRequest(MissingServletRequestParameterException ex) {
        log.error("全局异常处理 错误原因(错误请求参数) {}", ex);
        return ResultDTO.fail();
    }

    //405错误
    @ExceptionHandler({HttpRequestMethodNotSupportedException.class})
    @ResponseBody
    public ResultDTO request405(HttpRequestMethodNotSupportedException ex) {
        log.error("全局异常处理 错误原因(方法不允许) {}", ex);
        return ResultDTO.fail(405, ex.getMessage());
    }

    //406错误
    @ExceptionHandler({HttpMediaTypeNotAcceptableException.class})
    @ResponseBody
    public ResultDTO request406(HttpMediaTypeNotAcceptableException ex) {
        log.error("全局异常处理 错误原因(不能接受) {}", ex);
        return ResultDTO.fail(406, ex.getMessage());
    }

    //500错误
    @ExceptionHandler({ConversionNotSupportedException.class, HttpMessageNotWritableException.class})
    @ResponseBody
    public ResultDTO server500(RuntimeException ex) {
        log.error("全局异常处理 错误原因(服务器内部错误(运行时异常)) {}", ex);
        return ResultDTO.fail(500, ex.getMessage());
    }

    //500错误
    @ExceptionHandler(Exception.class)
    @ResponseBody
    public ResultDTO unknow500(RuntimeException ex) {
        log.error("全局异常处理 错误原因(服务器内部错误(非运行异常)) {}", ex);
        return ResultDTO.fail(500, ex.getMessage());
    }
}
