package com.chinamcloud.plugin.container;

import com.chinamcloud.plugin.code.*;
import lombok.extern.slf4j.Slf4j;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.*;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.zip.ZipEntry;

/**
 * Created by jyy on 2018/1/10.
 */
@Slf4j
public class PluginManager {

    /**
     * 插件所有的文件的路径
     */
    private final Path pluginDirectory;

    /**
     * 所有的插件的集合
     */
    private final Map<String, Plugin> plugins = new HashMap();

    /**
     * 被加载的插件的集合,这是一个可排序的集合,
     */
    private final Map<String, Plugin> pluginsLoaded = new TreeMap<>( String.CASE_INSENSITIVE_ORDER );


    /**
     * 依赖关系检查
     */
    private final Map<String, DependencyNode> dependencyNodeMap = new TreeMap<>(String.CASE_INSENSITIVE_ORDER);


    /**
     * 被禁止启动的插件
     */
    private final Map<String, Plugin> pluginUnload = new HashMap();


    /**
     *
     */
    private VersionManager versionManager = new DefaultVersionManager();


    /**
     *
     */
    private DependencyManager dependencyManager = new DefaultDependencyManager();





    /**
     * 禁止启动的插件列表
     */
    private List<String> list = Collections.EMPTY_LIST;


    public PluginManager(File pluginDir,List list) {
        this.pluginDirectory = pluginDir.toPath();
        this.list = list;
        findPlugin();
    }


    public void findPlugin()  {
        try{
            //插件所在目录
            Path pluginsDirectory = this.getPluginDirectory();
            if ( !Files.isDirectory( pluginsDirectory ) || !Files.isReadable( pluginsDirectory ) ) {
                log.error( "不能够加载插件目录目录不存在或者目录不可读 : {}", pluginsDirectory );
                return;
            }
            //过滤非jar的文件
            DirectoryStream<Path> ds = Files.newDirectoryStream( pluginsDirectory,
                    entry -> {
                        if (Files.isDirectory( entry ) ) {
                            return false;
                        }
                        String fileName = entry.getFileName().toString().toLowerCase();
                        return ( fileName.endsWith( ".jar" ));
                    });
            //遍历所有的文件
            ds.forEach(jarFile ->{
                try {
                    String fileName = jarFile.getFileName().toString();
                    jarFile.toUri();
                    log.debug("{} 被扫描到,开始进行组装插件",fileName);
                    JarFile file = new JarFile( jarFile.toFile() );
                    Plugin plugin = resolvePlugin(file);
                    if (plugin != null || dependencyManager.satisfies(plugin,this)){
                        plugin.setPath(jarFile);
                        if (list.contains(plugin.getPluginId()) || list.contains(plugin.getPluginDescriptor().getPluginName()) ){
                             log.info("{} 被扫描到,被禁止使用",fileName);
                        }else {
                            plugins.put(plugin.getPluginId(),plugin);
                        }

                    }
                } catch (IOException e) {
                    e.printStackTrace();
                }
            });

            //插件依赖检查
            plugins.forEach((pluginId,plugin)->{
                List<PluginDependency> dependencies = plugin.getPluginDescriptor().getDependencies();
                dependencies.forEach( dependency ->{

                });
            });

            plugins.forEach((pluginId,plugin)->{
                JarPluginLoader jarPluginLoader = new JarPluginLoader(this);
                jarPluginLoader.loadPlugin(plugin);

            });



        }catch (Exception e){
            e.printStackTrace();
        }


    }



    /**
     * 通过jar问价解析plugin文件信息
     * @param file
     * @return
     */
    private Plugin resolvePlugin(JarFile file){
        InputStream inputStream = null;
        try {
            //如果不存在plugin.xml
            ZipEntry fileEntry = file.getEntry("plugin.xml");
            if ( fileEntry == null ) {
                return null;
            }
            if (log.isDebugEnabled()){
                for (Enumeration<JarEntry> e = file.entries(); e.hasMoreElements(); ) {
                    JarEntry entry = e.nextElement();
                    System.out.println(entry.getName());
                }
            }
            inputStream = file.getInputStream(fileEntry);
            Properties prop = new Properties();
            prop.loadFromXML(inputStream);
            inputStream.close();
            String pluginId = prop.getProperty("pluginId");
            String pluginName = prop.getProperty("pluginName");
            String pluginDescription = prop.getProperty("pluginDescription");
            String version = prop.getProperty("version");
            String requires = prop.getProperty("requires");
            String provider = prop.getProperty("provider");
            String dependencies = prop.getProperty("dependencies");
            String readme = prop.getProperty("readme");
            String license = prop.getProperty("license");
            Plugin plugin = new Plugin();

            PluginDescriptor pluginDescriptor = new PluginDescriptor();
            pluginDescriptor.setPluginId(pluginId);
            pluginDescriptor.setPluginName(pluginName);
            pluginDescriptor.setPluginDescription(pluginDescription);
            pluginDescriptor.setVersion(version);
            pluginDescriptor.setRequires(requires);
            pluginDescriptor.setProvider(provider);
            pluginDescriptor.setLicense(license);
            pluginDescriptor.setDependencies(dependencies);
            pluginDescriptor.setReadMe(readme);
            plugin.setPluginDescriptor(pluginDescriptor);
            plugin.setPluginId(pluginId);
            return plugin;
        } catch (IOException e) {
            e.printStackTrace();
            log.error(e.getMessage());
        }
        return null;

    }

    public Path getPluginDirectory() {
        return pluginDirectory;
    }

    public Map<String, Plugin> getPluginsLoaded() {
        return pluginsLoaded;
    }

    public VersionManager getVersionManager() {
        return versionManager;
    }


    public void setVersionManager(VersionManager versionManager) {
        this.versionManager = versionManager;
    }

    public Map<String, Plugin> getPlugins() {
        return plugins;
    }

}
