package com.chinamcloud.plugin.interceptor;

import com.chinamcloud.plugin.annotation.Support;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ArrayUtils;
import org.springframework.aop.framework.AopProxyUtils;
import org.springframework.aop.support.AopUtils;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.config.BeanExpressionContext;
import org.springframework.beans.factory.config.BeanExpressionResolver;
import org.springframework.beans.factory.config.ConfigurableBeanFactory;
import org.springframework.context.expression.AnnotatedElementKey;
import org.springframework.core.annotation.AnnotationUtils;
import org.springframework.util.*;
import org.springframework.web.context.request.RequestScope;

import java.lang.reflect.Method;
import java.util.Collection;
import java.util.Collections;
/**
 * Created by jyy on 17/12/15.
 */
@Slf4j
public abstract  class PluginsAspectSupport  {

    /**
     * 从 Spring的 上下文中获得操作列表
     */
    private PluginOperationSource pluginsOperation;

    /**
     * 被注入Spring 的bean工厂
     */
    private  ConfigurableBeanFactory beanFactory;

    private  BeanExpressionContext expressionContext;


    public void setBeanFactory(ConfigurableBeanFactory beanFactory) throws BeansException {
        this.expressionContext =  (beanFactory != null ? new BeanExpressionContext((ConfigurableBeanFactory) beanFactory, new RequestScope()) : null);
        this.beanFactory = beanFactory;
    }

    /**
     *  执行拦截器的方法
     * @param invoker 方法 invoker
     * @param target  执行这个方法的
     * @param method  具体某个方法
     * @param args    方法翻出
     * @return 返回方法拦截器中的之后测结果集
     */
    protected Object execute(PluginOperationInvoker invoker, Object target, Method method, Object[] args) {
        //获得类的代理的class
        Class<?> targetClass = getTargetClass(target);
        //寻找这个类上所有的操作
        Collection<PluginsOperation> operations = getPluginsOperation().getPluginOperations(method, targetClass);
        if (!CollectionUtils.isEmpty(operations)) {
            return execute(invoker, new PluginsOperationContexts(operations, method, args, target, targetClass),new OperationContext(target,method,args));
        }
        return invoker.invoke(target, method, args);
    }


    /**
     *
     * @param invoker  方法 invoker
     * @param contexts 拦击器的集合
     * @return
     */
    private Object execute(final PluginOperationInvoker invoker,PluginsOperationContexts contexts,OperationContext operationContext){


        //前置方法拦截器
        Object[] objects = processBeforeFilter(contexts.get(SpiderBeforeOperation.class) ,operationContext);

        operationContext.setArgs(objects);
        //执行方法
        Object returnValue = invokeOperation(invoker,operationContext);

        //执行后置拦截器
        Object result = processAfterFilter(contexts.get(SpiderAfterOperation.class), returnValue);

        //执行事务监听
        return result;
    }


    private Object[]  processBeforeFilter(Collection<PluginsOperationContext> contexts,OperationContext operationContext) {

        //原始参数
        Object[] args = operationContext.getArgs();
        for (PluginsOperationContext context : contexts){
            context.setArgs(args); //将上一个拦截器的值赋给下个拦截器
            SpiderBeforeOperation operation = (SpiderBeforeOperation) context.getOperation();
            if (findSupport(operation.getObject())) {
                args = performBeforeFilter(context, operation);
            }
        }
        return args;
    }



    private Object processAfterFilter(Collection<PluginsOperationContext> contexts,Object result) {
        Object _result = result;
        for (PluginsOperationContext context : contexts) {
            SpiderAfterOperation operation = (SpiderAfterOperation) context.getOperation();
            if (findSupport(operation.getObject())) {
                Object o = performAfterFilter(context, operation, _result);
                _result = o;
            }
        }
        return _result;
    }




    private Object []  performBeforeFilter(PluginsOperationContext context, SpiderBeforeOperation operation) {

        Object bean = beanFactory.getBean(operation.gettargetClass());
        Object o = ReflectionUtils.invokeMethod(context.getOperation().getMethod(), bean, context.getArgs());

        //如果结果符合规范，则返回处理后的结果。否则返回原参数
        if (o instanceof  Object []){
           return (Object[]) o;
        }
        return context.getArgs();
    }



    private Object performAfterFilter(PluginsOperationContext context, SpiderAfterOperation operation,Object result) {
        Object bean = beanFactory.getBean(operation.gettargetClass());
        Object o = ReflectionUtils.invokeMethod(context.getOperation().getMethod(), bean, ArrayUtils.add(context.getArgs(), result));
        return o;
    }





    protected Object invokeOperation(PluginOperationInvoker invoker,OperationContext operationContext) {
        return invoker.invoke(operationContext.getTarget(),operationContext.getMethod(),operationContext.getArgs());
    }


    private class OperationContext{


        private  Object target;

        private  Method method;

        private  Object[] args;


        public OperationContext(Object target, Method method, Object[] args) {
            this.target = target;
            this.method = method;
            this.args = args;
        }

        public Object getTarget() {
            return target;
        }

        public void setTarget(Object target) {
            this.target = target;
        }

        public Method getMethod() {
            return method;
        }

        public void setMethod(Method method) {
            this.method = method;
        }

        public Object[] getArgs() {
            return args;
        }

        public void setArgs(Object[] args) {
            this.args = args;
        }
    }

    private  class PluginsOperationContexts{
        private final MultiValueMap<Class<? extends PluginsOperation>, PluginsOperationContext> contexts =
                new LinkedMultiValueMap<>();


        public PluginsOperationContexts(Collection<? extends PluginsOperation> operations, Method method,
                                        Object[] args, Object target, Class<?> targetClass) {

            for (PluginsOperation operation : operations) {
                this.contexts.add(operation.getClass(), getOperationContext(operation, method, args, target, targetClass));
            }
        }


        public Collection<PluginsOperationContext> get(Class<? extends PluginsOperation> operationClass) {
            Collection<PluginsOperationContext> result = this.contexts.get(operationClass);
            return (result != null ? result : Collections.emptyList());
        }


    }

    private PluginsOperationContext getOperationContext(
            PluginsOperation operation, Method method, Object[] args, Object target, Class<?> targetClass) {

        return new PluginsOperationContext(operation,method,targetClass,args,target);
    }

    private class  PluginsOperationContext {

        private  PluginsOperation operation;

        private  Method method;

        private  Class<?> targetClass;

        private  Object[] args;

        private  Object target;

        private  AnnotatedElementKey methodCacheKey;

        public PluginsOperationContext(PluginsOperation operation, Method method, Class<?> targetClass, Object[] args, Object target) {
            this.operation = operation;
            this.method = method;
            this.targetClass = targetClass;
            this.args = args;
            this.target = target;
            this.methodCacheKey = new AnnotatedElementKey(method,targetClass);
        }


        public PluginsOperation getOperation() {
            return operation;
        }

        public Method getMethod() {
            return method;
        }

        public Class<?> getTargetClass() {
            return targetClass;
        }

        public Object[] getArgs() {
            return args;
        }

        public Object getTarget() {
            return target;
        }

        public AnnotatedElementKey getMethodCacheKey() {
            return methodCacheKey;
        }


        public void setOperation(PluginsOperation operation) {
            this.operation = operation;
        }

        public void setMethod(Method method) {
            this.method = method;
        }

        public void setTargetClass(Class<?> targetClass) {
            this.targetClass = targetClass;
        }

        public void setArgs(Object[] args) {
            this.args = args;
        }

        public void setTarget(Object target) {
            this.target = target;
        }

        public void setMethodCacheKey(AnnotatedElementKey methodCacheKey) {
            this.methodCacheKey = methodCacheKey;
        }
    }

    private Class<?> getTargetClass(Object target) {
        Class<?> targetClass = AopProxyUtils.ultimateTargetClass(target);
        if (targetClass == null && target != null) {
            targetClass = target.getClass();
        }
        return targetClass;
    }

    public PluginOperationSource getPluginsOperation() {
        return pluginsOperation;
    }

    public void setPluginsOperation(PluginOperationSource pluginsOperation) {
        this.pluginsOperation = pluginsOperation;
    }

    protected boolean findSupport(Object obj){
        //检查接口
        if (obj instanceof Supported  ){
            Boolean result = ((Supported) obj).isSupport();
            return result;
        }
        //检查注解
        Support support = AnnotationUtils.findAnnotation(AopUtils.getTargetClass(obj), Support.class);
        if (support != null && (!StringUtils.isEmpty(support.value()))) {
            Object o = resolveStringValue(support.value());
            if (o != null && o instanceof  Boolean){
                return (Boolean) o;
            }
        }
        return true;
    }

    private Object resolveStringValue(String value) {
        value = wrapIfNecessary(value);
        ConfigurableBeanFactory configurableBeanFactory = (ConfigurableBeanFactory) beanFactory;
        BeanExpressionResolver exprResolver = configurableBeanFactory.getBeanExpressionResolver();
        if (exprResolver == null) {
            return null;
        }
        return exprResolver.evaluate(value, this.expressionContext);
    }


    private String wrapIfNecessary(String expression) {
        if (!expression.startsWith("#{")) {
            return "#{" + expression + "}";
        }
        return expression;
    }
}
