package com.chinamcloud.plugin.interceptor;

import com.chinamcloud.plugin.annotation.*;
import com.chinamcloud.plugin.configuration.PluginsComparator;
import lombok.extern.slf4j.Slf4j;
import org.springframework.aop.framework.AopProxyUtils;
import org.springframework.aop.framework.ProxyFactory;
import org.springframework.aop.support.AopUtils;
import org.springframework.beans.BeansException;
import org.springframework.beans.PropertyValues;
import org.springframework.beans.factory.annotation.AutowiredAnnotationBeanPostProcessor;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;
import org.springframework.beans.factory.config.InstantiationAwareBeanPostProcessorAdapter;
import org.springframework.core.MethodIntrospector;
import org.springframework.core.annotation.AnnotatedElementUtils;
import org.springframework.core.annotation.AnnotationUtils;
import org.springframework.util.ClassUtils;
import org.springframework.util.CollectionUtils;
import org.springframework.util.PatternMatchUtils;

import java.beans.PropertyDescriptor;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;

@Slf4j
public class SpiderBeanPostProcessor extends InstantiationAwareBeanPostProcessorAdapter implements PluginOperationSource{

    /**
     * 操作集合,缓存所有操作作类型
     */
    private final static Collection<PluginsOperation> NULL_CACHING_ATTRIBUTE = Collections.emptyList();
    /**
     * 本地缓存器
     */
    private final Map<Object, Collection<PluginsOperation>> attributeCache = new ConcurrentHashMap<>(1024);

    private   Collection<PluginsOperation> ops;

    private ConfigurableListableBeanFactory beanFactory;

    private  PluginInterceptor pluginInterceptor;


    public SpiderBeanPostProcessor(ConfigurableListableBeanFactory beanFactory) {
        this.beanFactory = beanFactory;
    }

    @Override
    public Object postProcessBeforeInitialization(Object o, String s) throws BeansException {
        return o;
    }

    @Override
    public Object postProcessAfterInitialization(Object bean, String s) throws BeansException {
        collectionOperation();
        Class  target = AopProxyUtils.ultimateTargetClass(bean);
        if (PluginPointcut.getClassFilter().matches(target)){
            Method[] methods = target.getMethods();
            for (Method method : methods){
                Collection<PluginsOperation> pluginOperations = getPluginOperations(method, target);
                if (CollectionUtils.isEmpty(pluginOperations)){
                    return bean;
                }else {
                    Class<?>[] interfaces = ClassUtils.getAllInterfaces(bean);
                    ProxyFactory proxyFactory = new ProxyFactory(interfaces);
                    proxyFactory.setTarget(bean);
                    proxyFactory.addAdvice(pluginInterceptor);
                    return proxyFactory.getProxy();
                }
            }
        }
        return bean;
    }

    @Override
    public PropertyValues postProcessPropertyValues(PropertyValues pvs, PropertyDescriptor[] pds, Object bean, String beanName) throws BeansException {

        Class  beanTargetClass = AopProxyUtils.ultimateTargetClass(bean);
        if (!CollectionUtils.isEmpty(AnnotationUtils.getRepeatableAnnotations(beanTargetClass, SpiderPlugin.class, SpiderPlugins.class))){
            AutowiredAnnotationBeanPostProcessor annotationBeanPostProcessor = beanFactory.getBean(AutowiredAnnotationBeanPostProcessor.class);
            annotationBeanPostProcessor.setBeanFactory(beanFactory);
            return  annotationBeanPostProcessor.postProcessPropertyValues(pvs, pds, bean, beanName);
        }
        return pvs;
    }


    public void collectionOperation()  {
        if (ops != null){
            return;
        }
        if (ops == null){
           ops = new HashSet<>();
        }
        //构造
        Map<String, Object> beansWithAnnotation = beanFactory.getBeansWithAnnotation(SpiderPlugin.class);
        beansWithAnnotation.forEach((name,bean)->{
            Class<?> beanTargetClass = AopUtils.getTargetClass(bean);
            //寻找所有前置拦截器
            Set<SpiderPlugin> spiderPlugin = AnnotationUtils.getRepeatableAnnotations(beanTargetClass, SpiderPlugin.class, SpiderPlugins.class);
            if (!CollectionUtils.isEmpty(spiderPlugin)){
                Map<Method, SpiderBeforeFilter> annotatedMethods = MethodIntrospector.selectMethods(beanTargetClass, new MethodIntrospector.MetadataLookup<SpiderBeforeFilter>() {
                    @Override
                    public SpiderBeforeFilter inspect(Method method) {
                        return AnnotatedElementUtils.getMergedAnnotation(method, SpiderBeforeFilter.class);
                    }});
                Integer order = PluginsComparator.instance.findOrder(bean);
                annotatedMethods.forEach((beforeMethod, beforeFilter) -> ops.add(parseBeforeAnnotation(bean,beanTargetClass, spiderPlugin,beforeFilter,order,beforeMethod)));

                Map<Method, SpiderListener> spiderListenerAnnotatedMethods = MethodIntrospector.selectMethods(beanTargetClass, new MethodIntrospector.MetadataLookup<SpiderListener>() {
                    @Override
                    public SpiderListener inspect(Method method) {
                        return AnnotatedElementUtils.getMergedAnnotation(method, SpiderListener.class);
                    }});
                spiderListenerAnnotatedMethods.forEach((beforeMethod, spiderListener) -> ops.add(parseListenerAnnotation(bean,beanTargetClass, spiderPlugin,spiderListener,order,beforeMethod)));

                Map<Method, SpiderAfterFilter> spiderAfterAnnotatedMethods = MethodIntrospector.selectMethods(beanTargetClass, new MethodIntrospector.MetadataLookup<SpiderAfterFilter>() {
                    @Override
                    public SpiderAfterFilter inspect(Method method) {
                        return AnnotatedElementUtils.getMergedAnnotation(method, SpiderAfterFilter.class);
                    }});
                spiderAfterAnnotatedMethods.forEach((beforeMethod, spiderAfter) -> ops.add(parseAfterAnnotation(bean,beanTargetClass, spiderPlugin,spiderAfter,order,beforeMethod)));

            }


        });
        pluginInterceptor = new PluginInterceptor();
        pluginInterceptor.setPluginsOperation(this);
        pluginInterceptor.setBeanFactory(beanFactory);
    }



    private  SpiderBeforeOperation parseBeforeAnnotation(Object target ,Class<?> targetClass, Set<SpiderPlugin> spiderBeforeFilter,SpiderBeforeFilter before, Integer order, Method method) {
        SpiderBeforeOperation.Builder builder = new SpiderBeforeOperation.Builder();
        builder.setOrder(order);
        builder.settargetClass(targetClass);
        builder.setMethod(method);
        builder.setObject(target);
        builder.setExpress(toExpress(spiderBeforeFilter));
        SpiderBeforeOperation op = builder.build();
        return op;
    }

    private  SpiderListenerOperation parseListenerAnnotation(Object target ,Class<?> targetClass, Set<SpiderPlugin> spiderBeforeFilter,SpiderListener listener, Integer order, Method method) {
        SpiderListenerOperation.Builder builder = new SpiderListenerOperation.Builder();
        builder.setOrder(order);
        builder.settargetClass(targetClass);
        builder.setMethod(method);
        builder.setObject(target);
        builder.setExpress(toExpress(spiderBeforeFilter));
        SpiderListenerOperation op = builder.build();
        return op;
    }


    private  SpiderAfterOperation parseAfterAnnotation(Object target ,Class<?> targetClass, Set<SpiderPlugin> spiderBeforeFilter,SpiderAfterFilter afterFilter, Integer order, Method method) {
        SpiderAfterOperation.Builder builder = new SpiderAfterOperation.Builder();
        builder.setOrder(order);
        builder.settargetClass(targetClass);
        builder.setMethod(method);
        builder.setObject(target);
        builder.setExpress(toExpress(spiderBeforeFilter));
        SpiderAfterOperation op = builder.build();
        return op;
    }


    private String toExpress(Set<SpiderPlugin> spiderBeforeFilter){
        if (CollectionUtils.isEmpty(spiderBeforeFilter)){
            return "";
        }
        StringBuilder stringBuilder = new StringBuilder();
        Iterator<SpiderPlugin> iterator = spiderBeforeFilter.iterator();
        while (iterator.hasNext()){
            stringBuilder.append(iterator.next().express());
            if (iterator.hasNext()){
                stringBuilder.append(" || ");
            }
        }
        return  new String(stringBuilder);
    }




    /**
     *  功能： <br />  method+targetClass key , <br />
     *  首先 检查缓存是否存在的对曾 key 的操作的方法,如果存在了从缓存各种读取,否则从content 上下文中读取。并保存缓存中
     * @param method  方法
     * @param targetClass 目标class
     * @return
     */
    public Collection<PluginsOperation> getPluginOperations(Method method, Class<?> targetClass) {

        Object cacheKey = getCacheKey(method, targetClass);
        Collection<PluginsOperation> cached = this.attributeCache.get(cacheKey);
        if (cached != null) {
            return (cached != NULL_CACHING_ATTRIBUTE ? cached : null);
        }
        else {
            Collection<PluginsOperation> cacheOps = computePluginOperations(method, targetClass);
            if (!CollectionUtils.isEmpty(cacheOps)) {
                log.debug("增加缓存 method : '" + method.getName() + "' 到: " + cacheOps);
                this.attributeCache.put(cacheKey, cacheOps);
            }
            else {
                this.attributeCache.put(cacheKey, NULL_CACHING_ATTRIBUTE);
            }
            return cacheOps;
        }
    }


    /**
     * 通过给定的method,和class 生成缓存key
     * @param method
     * @param targetClass
     * @return
     */
    protected Object getCacheKey(Method method, Class<?> targetClass) {
        return new MethodNameClassKey(method.getName(), targetClass,method.getReturnType(),method.getParameterTypes());
    }
    /**
     * 方法虚招拦截器方法
     * @param method 拦截的方法
     * @param targetClass 拦截类的class
     * @return
     */
    private Collection<PluginsOperation> computePluginOperations(Method method, Class<?> targetClass) {

        log.debug("扫描类注册 : " + targetClass);

        // 只允许public 方法才能实现注解
        if (!Modifier.isPublic(method.getModifiers())) {
            return null;
        }
        Collection<PluginsOperation> _ops = new HashSet<>();
        ops.forEach(pluginsOperation -> {
            if (PatternMatchUtils.simpleMatch(pluginsOperation.getExpress(),method.toString())){
                _ops.add(pluginsOperation);
            }

        });
        return _ops;
    }



}
