/*
 * Decompiled with CFR 0.152.
 */
package com.github.gavlyukovskiy.cloud.sleuth;

import brave.Span;
import brave.Tracer;
import com.github.gavlyukovskiy.cloud.sleuth.SleuthProperties;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;

class TracingListenerStrategy<CON, STMT, RS> {
    private final Map<CON, ConnectionInfo> openConnections = new ConcurrentHashMap<CON, ConnectionInfo>();
    private final Tracer tracer;
    private final List<SleuthProperties.TraceType> traceTypes;

    TracingListenerStrategy(Tracer tracer, List<SleuthProperties.TraceType> traceTypes) {
        this.tracer = tracer;
        this.traceTypes = traceTypes;
    }

    void beforeGetConnection(CON connectionKey, String dataSourceName) {
        SpanWithScope spanWithScope = null;
        if (this.traceTypes.contains((Object)SleuthProperties.TraceType.CONNECTION)) {
            Span connectionSpan = this.tracer.nextSpan().name("jdbc:/" + dataSourceName + "/connection");
            connectionSpan.remoteServiceName(dataSourceName);
            connectionSpan.kind(Span.Kind.CLIENT);
            connectionSpan.start();
            spanWithScope = new SpanWithScope(connectionSpan, this.tracer.withSpanInScope(connectionSpan));
        }
        ConnectionInfo connectionInfo = new ConnectionInfo(spanWithScope);
        this.openConnections.put(connectionKey, connectionInfo);
    }

    void afterGetConnection(CON connectionKey, Throwable t) {
        if (t != null) {
            ConnectionInfo connectionInfo = this.openConnections.remove(connectionKey);
            connectionInfo.getSpan().ifPresent(connectionSpan -> {
                connectionSpan.getSpan().error(t);
                connectionSpan.finish();
            });
        }
    }

    void beforeQuery(CON connectionKey, STMT statementKey, String dataSourceName) {
        SpanWithScope spanWithScope = null;
        if (this.traceTypes.contains((Object)SleuthProperties.TraceType.QUERY)) {
            Span statementSpan = this.tracer.nextSpan().name("jdbc:/" + dataSourceName + "/query");
            statementSpan.remoteServiceName(dataSourceName);
            statementSpan.kind(Span.Kind.CLIENT);
            statementSpan.start();
            spanWithScope = new SpanWithScope(statementSpan, this.tracer.withSpanInScope(statementSpan));
        }
        StatementInfo statementInfo = new StatementInfo(spanWithScope);
        ConnectionInfo connectionInfo = this.openConnections.get(connectionKey);
        if (connectionInfo == null) {
            return;
        }
        connectionInfo.getNestedStatements().put(statementKey, statementInfo);
    }

    void addQueryRowCount(CON connectionKey, STMT statementKey, int rowCount) {
        ConnectionInfo connectionInfo = this.openConnections.get(connectionKey);
        if (connectionInfo == null) {
            return;
        }
        StatementInfo statementInfo = connectionInfo.getNestedStatements().get(statementKey);
        statementInfo.getSpan().ifPresent(statementSpan -> statementSpan.getSpan().tag("row-count", String.valueOf(rowCount)));
    }

    void afterQuery(CON connectionKey, STMT statementKey, String sql, Throwable t) {
        ConnectionInfo connectionInfo = this.openConnections.get(connectionKey);
        if (connectionInfo == null) {
            return;
        }
        StatementInfo statementInfo = connectionInfo.getNestedStatements().get(statementKey);
        statementInfo.getSpan().ifPresent(statementSpan -> {
            statementSpan.getSpan().tag("sql", sql);
            if (t != null) {
                statementSpan.getSpan().error(t);
            }
            statementSpan.finish();
        });
    }

    void beforeResultSetNext(CON connectionKey, STMT statementKey, RS resultSetKey, String dataSourceName) {
        if (!this.traceTypes.contains((Object)SleuthProperties.TraceType.FETCH)) {
            return;
        }
        ConnectionInfo connectionInfo = this.openConnections.get(connectionKey);
        if (connectionInfo == null) {
            return;
        }
        if (connectionInfo.getNestedResultSetSpans().containsKey(resultSetKey)) {
            return;
        }
        Span resultSetSpan = this.tracer.nextSpan().name("jdbc:/" + dataSourceName + "/fetch");
        resultSetSpan.remoteServiceName(dataSourceName);
        resultSetSpan.kind(Span.Kind.CLIENT);
        resultSetSpan.start();
        SpanWithScope spanWithScope = new SpanWithScope(resultSetSpan, this.tracer.withSpanInScope(resultSetSpan));
        connectionInfo.getNestedResultSetSpans().put(resultSetKey, spanWithScope);
        StatementInfo statementInfo = connectionInfo.getNestedStatements().get(statementKey);
        if (statementInfo != null) {
            statementInfo.getNestedResultSetSpans().put(resultSetKey, spanWithScope);
        }
    }

    void afterResultSetClose(CON connectionKey, RS resultSetKey, int rowCount, Throwable t) {
        ConnectionInfo connectionInfo = this.openConnections.get(connectionKey);
        if (connectionInfo == null) {
            return;
        }
        SpanWithScope resultSetSpan = connectionInfo.getNestedResultSetSpans().remove(resultSetKey);
        if (resultSetSpan == null) {
            return;
        }
        if (rowCount != -1) {
            resultSetSpan.getSpan().tag("row-count", String.valueOf(rowCount));
        }
        if (t != null) {
            resultSetSpan.getSpan().error(t);
        }
        resultSetSpan.finish();
    }

    void afterStatementClose(CON connectionKey, STMT statementKey) {
        ConnectionInfo connectionInfo = this.openConnections.get(connectionKey);
        if (connectionInfo == null) {
            return;
        }
        StatementInfo statementInfo = connectionInfo.getNestedStatements().remove(statementKey);
        if (statementInfo != null) {
            statementInfo.getNestedResultSetSpans().forEach((resultSetKey, span) -> {
                connectionInfo.getNestedResultSetSpans().remove(resultSetKey);
                span.finish();
            });
            statementInfo.getNestedResultSetSpans().clear();
        }
    }

    void afterCommit(CON connectionKey, Throwable t) {
        ConnectionInfo connectionInfo = this.openConnections.get(connectionKey);
        if (connectionInfo == null) {
            return;
        }
        connectionInfo.getSpan().ifPresent(connectionSpan -> {
            if (t != null) {
                connectionSpan.getSpan().error(t);
            }
            connectionSpan.getSpan().annotate("commit");
        });
    }

    void afterRollback(CON connectionKey, Throwable t) {
        ConnectionInfo connectionInfo = this.openConnections.get(connectionKey);
        if (connectionInfo == null) {
            return;
        }
        connectionInfo.getSpan().ifPresent(connectionSpan -> {
            if (t != null) {
                connectionSpan.getSpan().error(t);
            } else {
                connectionSpan.getSpan().tag("error", "Transaction rolled back");
            }
            connectionSpan.getSpan().annotate("rollback");
        });
    }

    void afterConnectionClose(CON connectionKey, Throwable t) {
        ConnectionInfo connectionInfo = this.openConnections.remove(connectionKey);
        if (connectionInfo == null) {
            return;
        }
        connectionInfo.getNestedResultSetSpans().values().forEach(SpanWithScope::finish);
        connectionInfo.getNestedStatements().values().forEach(statementInfo -> statementInfo.getSpan().ifPresent(SpanWithScope::finish));
        connectionInfo.getSpan().ifPresent(connectionSpan -> {
            if (t != null) {
                connectionSpan.getSpan().error(t);
            }
            connectionSpan.finish();
        });
    }

    private static class SpanWithScope {
        private final Span span;
        private final Tracer.SpanInScope spanInScope;

        private SpanWithScope(Span span, Tracer.SpanInScope spanInScope) {
            this.span = span;
            this.spanInScope = spanInScope;
        }

        Span getSpan() {
            return this.span;
        }

        void finish() {
            this.spanInScope.close();
            this.span.finish();
        }
    }

    private class StatementInfo {
        private final SpanWithScope span;
        private final Map<RS, SpanWithScope> nestedResultSetSpans = new ConcurrentHashMap();

        private StatementInfo(SpanWithScope span) {
            this.span = span;
        }

        Optional<SpanWithScope> getSpan() {
            return Optional.ofNullable(this.span);
        }

        Map<RS, SpanWithScope> getNestedResultSetSpans() {
            return this.nestedResultSetSpans;
        }
    }

    private class ConnectionInfo {
        private final SpanWithScope span;
        private final Map<STMT, StatementInfo> nestedStatements = new ConcurrentHashMap();
        private final Map<RS, SpanWithScope> nestedResultSetSpans = new ConcurrentHashMap();

        private ConnectionInfo(SpanWithScope span) {
            this.span = span;
        }

        Optional<SpanWithScope> getSpan() {
            return Optional.ofNullable(this.span);
        }

        Map<STMT, StatementInfo> getNestedStatements() {
            return this.nestedStatements;
        }

        Map<RS, SpanWithScope> getNestedResultSetSpans() {
            return this.nestedResultSetSpans;
        }
    }
}

