/*
 * Decompiled with CFR 0.152.
 */
package org.apache.fory.memory;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.Arrays;
import org.apache.fory.io.AbstractStreamReader;
import org.apache.fory.io.ForyStreamReader;
import org.apache.fory.memory.BoundsChecking;
import org.apache.fory.memory.ByteBufferUtil;
import org.apache.fory.memory.Platform;
import org.apache.fory.util.Preconditions;
import sun.misc.Unsafe;

public final class MemoryBuffer {
    public static final int BUFFER_GROW_STEP_THRESHOLD = 0x6400000;
    private static final Unsafe UNSAFE = Platform.UNSAFE;
    private static final boolean LITTLE_ENDIAN = ByteOrder.nativeOrder() == ByteOrder.LITTLE_ENDIAN;
    private byte[] heapMemory;
    private int heapOffset;
    private ByteBuffer offHeapBuffer;
    private long address;
    private long addressLimit;
    private int size;
    private int readerIndex;
    private int writerIndex;
    private final ForyStreamReader streamReader;
    private static final long HALF_MAX_INT_VALUE = 0x3FFFFFFFL;
    private static final long HALF_MIN_INT_VALUE = -1073741824L;
    private static final byte BIG_LONG_FLAG = 1;

    private MemoryBuffer(byte[] buffer, int offset, int length) {
        this(buffer, offset, length, null);
    }

    private MemoryBuffer(byte[] buffer, int offset, int length, ForyStreamReader streamReader) {
        Preconditions.checkArgument(offset >= 0 && length >= 0);
        if (offset + length > buffer.length) {
            throw new IllegalArgumentException(String.format("%d exceeds buffer size %d", offset + length, buffer.length));
        }
        this.initHeapBuffer(buffer, offset, length);
        this.streamReader = streamReader != null ? streamReader : new BoundChecker();
    }

    private MemoryBuffer(long offHeapAddress, int size, ByteBuffer offHeapBuffer) {
        this(offHeapAddress, size, offHeapBuffer, null);
    }

    private MemoryBuffer(long offHeapAddress, int size, ByteBuffer offHeapBuffer, ForyStreamReader streamReader) {
        this.initDirectBuffer(offHeapAddress, size, offHeapBuffer);
        this.streamReader = streamReader != null ? streamReader : new BoundChecker();
    }

    public void initDirectBuffer(long offHeapAddress, int size, ByteBuffer offHeapBuffer) {
        this.offHeapBuffer = offHeapBuffer;
        if (offHeapAddress <= 0L) {
            throw new IllegalArgumentException("negative pointer or size");
        }
        if (offHeapAddress >= 9223372034707292160L) {
            throw new IllegalArgumentException("Buffer initialized with too large address: " + offHeapAddress + " ; Max allowed address is " + 0x7FFFFFFF7FFFFFFFL);
        }
        this.heapMemory = null;
        this.address = offHeapAddress;
        this.addressLimit = this.address + (long)size;
        this.size = size;
    }

    public void initHeapBuffer(byte[] buffer, int offset, int length) {
        long startPos;
        if (buffer == null) {
            throw new NullPointerException("buffer");
        }
        this.heapMemory = buffer;
        this.heapOffset = offset;
        this.address = startPos = (long)(Platform.BYTE_ARRAY_OFFSET + offset);
        this.size = length;
        this.addressLimit = startPos + (long)length;
    }

    public int size() {
        return this.size;
    }

    public void increaseSize(int diff) {
        this.addressLimit = this.address + (long)(this.size += diff);
    }

    public boolean isOffHeap() {
        return this.heapMemory == null;
    }

    public boolean isHeapFullyWriteable() {
        return this.heapMemory != null && this.heapOffset == 0;
    }

    public byte[] getHeapMemory() {
        return this.heapMemory;
    }

    public ByteBuffer getOffHeapBuffer() {
        if (this.offHeapBuffer != null) {
            return this.offHeapBuffer;
        }
        throw new IllegalStateException("Memory buffer does not represent off heap ByteBuffer");
    }

    public byte[] getArray() {
        if (this.heapMemory != null) {
            return this.heapMemory;
        }
        throw new IllegalStateException("Memory buffer does not represent heap memory");
    }

    public long getAddress() {
        if (this.heapMemory == null) {
            return this.address;
        }
        throw new IllegalStateException("Memory buffer does not represent off heap memory");
    }

    public long getUnsafeAddress() {
        return this.address;
    }

    private void checkPosition(long index, long pos, long length) {
        if (BoundsChecking.BOUNDS_CHECKING_ENABLED && (index < 0L || pos > this.addressLimit - length)) {
            this.throwOOBException();
        }
    }

    public void get(int index, byte[] dst) {
        this.get(index, dst, 0, dst.length);
    }

    public void get(int index, byte[] dst, int offset, int length) {
        byte[] heapMemory = this.heapMemory;
        if (heapMemory != null) {
            System.arraycopy(heapMemory, this.heapOffset + index, dst, offset, length);
        } else {
            long pos = this.address + (long)index;
            if (((long)(index | offset | length | offset + length | dst.length - (offset + length)) | this.addressLimit - (long)length - pos) < 0L) {
                this.throwOOBException();
            }
            Platform.copyMemory(null, pos, dst, Platform.BYTE_ARRAY_OFFSET + offset, length);
        }
    }

    public void get(int offset, ByteBuffer target, int numBytes) {
        if ((offset | numBytes | offset + numBytes) < 0) {
            this.throwOOBException();
        }
        if (target.remaining() < numBytes) {
            this.throwOOBException();
        }
        if (target.isReadOnly()) {
            throw new IllegalArgumentException("read only buffer");
        }
        int targetPos = target.position();
        if (target.isDirect()) {
            long targetAddr = ByteBufferUtil.getAddress(target) + (long)targetPos;
            long sourceAddr = this.address + (long)offset;
            if (sourceAddr <= this.addressLimit - (long)numBytes) {
                Platform.copyMemory(this.heapMemory, sourceAddr, null, targetAddr, numBytes);
            } else {
                this.throwOOBException();
            }
        } else {
            assert (target.hasArray());
            this.get(offset, target.array(), targetPos + target.arrayOffset(), numBytes);
        }
        ByteBufferUtil.position(target, targetPos + numBytes);
    }

    public void put(int offset, ByteBuffer source, int numBytes) {
        int remaining = source.remaining();
        if ((offset | numBytes | offset + numBytes | remaining - numBytes) < 0) {
            this.throwOOBException();
        }
        int sourcePos = source.position();
        if (source.isDirect()) {
            long sourceAddr = ByteBufferUtil.getAddress(source) + (long)sourcePos;
            long targetAddr = this.address + (long)offset;
            if (targetAddr <= this.addressLimit - (long)numBytes) {
                Platform.copyMemory(null, sourceAddr, this.heapMemory, targetAddr, numBytes);
            } else {
                this.throwOOBException();
            }
        } else {
            assert (source.hasArray());
            this.put(offset, source.array(), sourcePos + source.arrayOffset(), numBytes);
        }
        ByteBufferUtil.position(source, sourcePos + numBytes);
    }

    public void put(int index, byte[] src) {
        this.put(index, src, 0, src.length);
    }

    public void put(int index, byte[] src, int offset, int length) {
        byte[] heapMemory = this.heapMemory;
        if (heapMemory != null) {
            System.arraycopy(src, offset, heapMemory, this.heapOffset + index, length);
        } else {
            long pos = this.address + (long)index;
            if (((long)(index | offset | length | offset + length | src.length - (offset + length)) | this.addressLimit - (long)length - pos) < 0L) {
                this.throwOOBException();
            }
            long arrayAddress = Platform.BYTE_ARRAY_OFFSET + offset;
            Platform.copyMemory(src, arrayAddress, null, pos, length);
        }
    }

    public byte getByte(int index) {
        long pos = this.address + (long)index;
        this.checkPosition(index, pos, 1L);
        return UNSAFE.getByte(this.heapMemory, pos);
    }

    public void putByte(int index, int b) {
        long pos = this.address + (long)index;
        this.checkPosition(index, pos, 1L);
        UNSAFE.putByte(this.heapMemory, pos, (byte)b);
    }

    public void putByte(int index, byte b) {
        long pos = this.address + (long)index;
        this.checkPosition(index, pos, 1L);
        UNSAFE.putByte(this.heapMemory, pos, b);
    }

    public boolean getBoolean(int index) {
        long pos = this.address + (long)index;
        this.checkPosition(index, pos, 1L);
        return UNSAFE.getByte(this.heapMemory, pos) != 0;
    }

    public void putBoolean(int index, boolean value) {
        UNSAFE.putByte(this.heapMemory, this.address + (long)index, value ? (byte)1 : 0);
    }

    public char getChar(int index) {
        long pos = this.address + (long)index;
        this.checkPosition(index, pos, 2L);
        char c = UNSAFE.getChar(this.heapMemory, pos);
        return LITTLE_ENDIAN ? c : Character.reverseBytes(c);
    }

    public void putChar(int index, char value) {
        long pos = this.address + (long)index;
        this.checkPosition(index, pos, 2L);
        if (!LITTLE_ENDIAN) {
            value = Character.reverseBytes(value);
        }
        UNSAFE.putChar(this.heapMemory, pos, value);
    }

    public short getInt16(int index) {
        long pos = this.address + (long)index;
        this.checkPosition(index, pos, 2L);
        short v = UNSAFE.getShort(this.heapMemory, pos);
        return LITTLE_ENDIAN ? v : Short.reverseBytes(v);
    }

    public void putInt16(int index, short value) {
        long pos = this.address + (long)index;
        this.checkPosition(index, pos, 2L);
        if (!LITTLE_ENDIAN) {
            value = Short.reverseBytes(value);
        }
        UNSAFE.putShort(this.heapMemory, pos, value);
    }

    public int getInt32(int index) {
        long pos = this.address + (long)index;
        this.checkPosition(index, pos, 4L);
        int v = UNSAFE.getInt(this.heapMemory, pos);
        return LITTLE_ENDIAN ? v : Integer.reverseBytes(v);
    }

    public void putInt32(int index, int value) {
        long pos = this.address + (long)index;
        this.checkPosition(index, pos, 4L);
        if (!LITTLE_ENDIAN) {
            value = Integer.reverseBytes(value);
        }
        UNSAFE.putInt(this.heapMemory, pos, value);
    }

    private int _unsafeGetInt32(int index) {
        int v = UNSAFE.getInt(this.heapMemory, this.address + (long)index);
        return LITTLE_ENDIAN ? v : Integer.reverseBytes(v);
    }

    public void _unsafePutInt32(int index, int value) {
        if (!LITTLE_ENDIAN) {
            value = Integer.reverseBytes(value);
        }
        UNSAFE.putInt(this.heapMemory, this.address + (long)index, value);
    }

    public long getInt64(int index) {
        long pos = this.address + (long)index;
        this.checkPosition(index, pos, 8L);
        long v = UNSAFE.getLong(this.heapMemory, pos);
        return LITTLE_ENDIAN ? v : Long.reverseBytes(v);
    }

    public void putInt64(int index, long value) {
        long pos = this.address + (long)index;
        this.checkPosition(index, pos, 8L);
        if (!LITTLE_ENDIAN) {
            value = Long.reverseBytes(value);
        }
        UNSAFE.putLong(this.heapMemory, pos, value);
    }

    long _unsafeGetInt64(int index) {
        long v = UNSAFE.getLong(this.heapMemory, this.address + (long)index);
        return LITTLE_ENDIAN ? v : Long.reverseBytes(v);
    }

    private void _unsafePutInt64(int index, long value) {
        if (!LITTLE_ENDIAN) {
            value = Long.reverseBytes(value);
        }
        UNSAFE.putLong(this.heapMemory, this.address + (long)index, value);
    }

    public float getFloat32(int index) {
        long pos = this.address + (long)index;
        this.checkPosition(index, pos, 8L);
        int v = UNSAFE.getInt(this.heapMemory, pos);
        if (!LITTLE_ENDIAN) {
            v = Integer.reverseBytes(v);
        }
        return Float.intBitsToFloat(v);
    }

    public void putFloat32(int index, float value) {
        long pos = this.address + (long)index;
        this.checkPosition(index, pos, 4L);
        int v = Float.floatToRawIntBits(value);
        if (!LITTLE_ENDIAN) {
            v = Integer.reverseBytes(v);
        }
        UNSAFE.putInt(this.heapMemory, pos, v);
    }

    public double getFloat64(int index) {
        long pos = this.address + (long)index;
        this.checkPosition(index, pos, 8L);
        long v = UNSAFE.getLong(this.heapMemory, pos);
        if (!LITTLE_ENDIAN) {
            v = Long.reverseBytes(v);
        }
        return Double.longBitsToDouble(v);
    }

    public void putFloat64(int index, double value) {
        long pos = this.address + (long)index;
        this.checkPosition(index, pos, 8L);
        long v = Double.doubleToRawLongBits(value);
        if (!LITTLE_ENDIAN) {
            v = Long.reverseBytes(v);
        }
        UNSAFE.putLong(this.heapMemory, pos, v);
    }

    private void throwOOBException() {
        throw new IndexOutOfBoundsException(String.format("size: %d, address %s, addressLimit %d", this.size, this.address, this.addressLimit));
    }

    public int writerIndex() {
        return this.writerIndex;
    }

    public void writerIndex(int writerIndex) {
        if (writerIndex < 0 || writerIndex > this.size) {
            this.throwOOBExceptionForWriteIndex(writerIndex);
        }
        this.writerIndex = writerIndex;
    }

    private void throwOOBExceptionForWriteIndex(int writerIndex) {
        throw new IndexOutOfBoundsException(String.format("writerIndex: %d (expected: 0 <= writerIndex <= size(%d))", writerIndex, this.size));
    }

    public void _unsafeWriterIndex(int writerIndex) {
        this.writerIndex = writerIndex;
    }

    public int _unsafeHeapWriterIndex() {
        return this.writerIndex + this.heapOffset;
    }

    public long _unsafeWriterAddress() {
        return this.address + (long)this.writerIndex;
    }

    public void _increaseWriterIndexUnsafe(int diff) {
        this.writerIndex += diff;
    }

    public void increaseWriterIndex(int diff) {
        int writerIdx = this.writerIndex + diff;
        this.ensure(writerIdx);
        this.writerIndex = writerIdx;
    }

    public void writeBoolean(boolean value) {
        int writerIdx = this.writerIndex;
        int newIdx = writerIdx + 1;
        this.ensure(newIdx);
        long pos = this.address + (long)writerIdx;
        UNSAFE.putByte(this.heapMemory, pos, (byte)(value ? 1 : 0));
        this.writerIndex = newIdx;
    }

    public void _unsafeWriteByte(byte value) {
        int writerIdx = this.writerIndex;
        int newIdx = writerIdx + 1;
        long pos = this.address + (long)writerIdx;
        UNSAFE.putByte(this.heapMemory, pos, value);
        this.writerIndex = newIdx;
    }

    public void writeByte(byte value) {
        int writerIdx = this.writerIndex;
        int newIdx = writerIdx + 1;
        this.ensure(newIdx);
        long pos = this.address + (long)writerIdx;
        UNSAFE.putByte(this.heapMemory, pos, value);
        this.writerIndex = newIdx;
    }

    public void writeByte(int value) {
        this.writeByte((byte)value);
    }

    public void writeChar(char value) {
        int writerIdx = this.writerIndex;
        int newIdx = writerIdx + 2;
        this.ensure(newIdx);
        long pos = this.address + (long)writerIdx;
        if (!LITTLE_ENDIAN) {
            value = Character.reverseBytes(value);
        }
        UNSAFE.putChar(this.heapMemory, pos, value);
        this.writerIndex = newIdx;
    }

    public void writeInt16(short value) {
        int writerIdx = this.writerIndex;
        int newIdx = writerIdx + 2;
        this.ensure(newIdx);
        if (!LITTLE_ENDIAN) {
            value = Short.reverseBytes(value);
        }
        UNSAFE.putShort(this.heapMemory, this.address + (long)writerIdx, value);
        this.writerIndex = newIdx;
    }

    public void writeInt32(int value) {
        int writerIdx = this.writerIndex;
        int newIdx = writerIdx + 4;
        this.ensure(newIdx);
        if (!LITTLE_ENDIAN) {
            value = Integer.reverseBytes(value);
        }
        UNSAFE.putInt(this.heapMemory, this.address + (long)writerIdx, value);
        this.writerIndex = newIdx;
    }

    public void writeInt64(long value) {
        int writerIdx = this.writerIndex;
        int newIdx = writerIdx + 8;
        this.ensure(newIdx);
        if (!LITTLE_ENDIAN) {
            value = Long.reverseBytes(value);
        }
        UNSAFE.putLong(this.heapMemory, this.address + (long)writerIdx, value);
        this.writerIndex = newIdx;
    }

    public void writeFloat32(float value) {
        int writerIdx = this.writerIndex;
        int newIdx = writerIdx + 4;
        this.ensure(newIdx);
        int v = Float.floatToRawIntBits(value);
        if (!LITTLE_ENDIAN) {
            v = Integer.reverseBytes(v);
        }
        UNSAFE.putInt(this.heapMemory, this.address + (long)writerIdx, v);
        this.writerIndex = newIdx;
    }

    public void writeFloat64(double value) {
        int writerIdx = this.writerIndex;
        int newIdx = writerIdx + 8;
        this.ensure(newIdx);
        long v = Double.doubleToRawLongBits(value);
        if (!LITTLE_ENDIAN) {
            v = Long.reverseBytes(v);
        }
        UNSAFE.putLong(this.heapMemory, this.address + (long)writerIdx, v);
        this.writerIndex = newIdx;
    }

    public int writeVarInt32(int v) {
        this.ensure(this.writerIndex + 8);
        int varintBytes = this._unsafePutVarUint36Small(this.writerIndex, (long)v << 1 ^ (long)(v >> 31));
        this.writerIndex += varintBytes;
        return varintBytes;
    }

    public int _unsafeWriteVarInt32(int v) {
        int varintBytes = this._unsafePutVarUint36Small(this.writerIndex, (long)v << 1 ^ (long)(v >> 31));
        this.writerIndex += varintBytes;
        return varintBytes;
    }

    public int writeVarUint32(int v) {
        this.ensure(this.writerIndex + 8);
        int varintBytes = this._unsafePutVarUint36Small(this.writerIndex, v);
        this.writerIndex += varintBytes;
        return varintBytes;
    }

    public int _unsafeWriteVarUint32(int v) {
        int varintBytes = this._unsafePutVarUint36Small(this.writerIndex, v);
        this.writerIndex += varintBytes;
        return varintBytes;
    }

    public int writeVarUint32Small7(int value) {
        this.ensure(this.writerIndex + 8);
        if (value >>> 7 == 0) {
            UNSAFE.putByte(this.heapMemory, this.address + (long)this.writerIndex++, (byte)value);
            return 1;
        }
        return this.continueWriteVarUint32Small7(value);
    }

    private int continueWriteVarUint32Small7(int value) {
        long encoded = value & 0x7F;
        encoded |= (long)((value & 0x3F80) << 1 | 0x80);
        int writerIdx = this.writerIndex;
        if (value >>> 14 == 0) {
            this._unsafePutInt32(writerIdx, (int)encoded);
            this.writerIndex += 2;
            return 2;
        }
        int diff = this.continuePutVarInt36(writerIdx, encoded, value);
        this.writerIndex += diff;
        return diff;
    }

    public int _unsafePutVarUint36Small(int index, long value) {
        long encoded = value & 0x7FL;
        if (value >>> 7 == 0L) {
            UNSAFE.putByte(this.heapMemory, this.address + (long)index, (byte)value);
            return 1;
        }
        encoded |= (value & 0x3F80L) << 1 | 0x80L;
        if (value >>> 14 == 0L) {
            this._unsafePutInt32(index, (int)encoded);
            return 2;
        }
        return this.continuePutVarInt36(index, encoded, value);
    }

    private int continuePutVarInt36(int index, long encoded, long value) {
        encoded |= (value & 0x1FC000L) << 2 | 0x8000L;
        if (value >>> 21 == 0L) {
            this._unsafePutInt32(index, (int)encoded);
            return 3;
        }
        encoded |= (value & 0xFE00000L) << 3 | 0x800000L;
        if (value >>> 28 == 0L) {
            this._unsafePutInt32(index, (int)encoded);
            return 4;
        }
        this._unsafePutInt64(index, encoded |= (value & 0xFF0000000L) << 4 | 0x80000000L);
        return 5;
    }

    public int writeVarUint32Aligned(int value) {
        if (value >>> 6 == 0) {
            return this.writeVarUint32Aligned1(value);
        }
        if (value >>> 12 == 0) {
            return this.writeVarUint32Aligned2(value);
        }
        if (value >>> 18 == 0) {
            return this.writeVarUint32Aligned3(value);
        }
        if (value >>> 24 == 0) {
            return this.writeVarUint32Aligned4(value);
        }
        if (value >>> 30 == 0) {
            return this.writeVarUint32Aligned5(value);
        }
        return this.writeVarUint32Aligned6(value);
    }

    private int writeVarUint32Aligned1(int value) {
        int writerIdx = this.writerIndex;
        int numPaddingBytes = 4 - writerIdx % 4;
        this.ensure(writerIdx + 5);
        int first = value & 0x3F;
        long pos = this.address + (long)writerIdx;
        if (numPaddingBytes == 1) {
            UNSAFE.putByte(this.heapMemory, pos, (byte)(first | 0x40));
            this.writerIndex = writerIdx + 1;
            return 1;
        }
        UNSAFE.putByte(this.heapMemory, pos, (byte)first);
        UNSAFE.putInt(this.heapMemory, pos + 1L, 0);
        UNSAFE.putByte(this.heapMemory, pos + (long)numPaddingBytes - 1L, (byte)64);
        this.writerIndex = writerIdx + numPaddingBytes;
        return numPaddingBytes;
    }

    private int writeVarUint32Aligned2(int value) {
        int writerIdx = this.writerIndex;
        int numPaddingBytes = 4 - writerIdx % 4;
        this.ensure(writerIdx + 6);
        int first = value & 0x3F;
        long pos = this.address + (long)writerIdx;
        UNSAFE.putByte(this.heapMemory, pos, (byte)(first | 0x80));
        if (numPaddingBytes == 2) {
            UNSAFE.putByte(this.heapMemory, pos + 1L, (byte)(value >>> 6 | 0x40));
            this.writerIndex = writerIdx + 2;
            return 2;
        }
        UNSAFE.putByte(this.heapMemory, pos + 1L, (byte)(value >>> 6));
        UNSAFE.putInt(this.heapMemory, pos + 2L, 0);
        if (numPaddingBytes > 2) {
            UNSAFE.putByte(this.heapMemory, pos + (long)numPaddingBytes - 1L, (byte)64);
            this.writerIndex = writerIdx + numPaddingBytes;
            return numPaddingBytes;
        }
        UNSAFE.putByte(this.heapMemory, pos + 4L, (byte)64);
        this.writerIndex = writerIdx + numPaddingBytes + 4;
        return numPaddingBytes + 4;
    }

    private int writeVarUint32Aligned3(int value) {
        int writerIdx = this.writerIndex;
        int numPaddingBytes = 4 - writerIdx % 4;
        this.ensure(writerIdx + 7);
        int first = value & 0x3F;
        long pos = this.address + (long)writerIdx;
        UNSAFE.putByte(this.heapMemory, pos, (byte)(first | 0x80));
        UNSAFE.putByte(this.heapMemory, pos + 1L, (byte)(value >>> 6 | 0x80));
        if (numPaddingBytes == 3) {
            UNSAFE.putByte(this.heapMemory, pos + 2L, (byte)(value >>> 12 | 0x40));
            this.writerIndex = writerIdx + 3;
            return 3;
        }
        UNSAFE.putByte(this.heapMemory, pos + 2L, (byte)(value >>> 12));
        UNSAFE.putInt(this.heapMemory, pos + 3L, 0);
        if (numPaddingBytes == 4) {
            UNSAFE.putByte(this.heapMemory, pos + (long)numPaddingBytes - 1L, (byte)64);
            this.writerIndex = writerIdx + numPaddingBytes;
            return numPaddingBytes;
        }
        UNSAFE.putByte(this.heapMemory, pos + (long)numPaddingBytes + 3L, (byte)64);
        this.writerIndex = writerIdx + numPaddingBytes + 4;
        return numPaddingBytes + 4;
    }

    private int writeVarUint32Aligned4(int value) {
        int writerIdx = this.writerIndex;
        int numPaddingBytes = 4 - writerIdx % 4;
        this.ensure(writerIdx + 8);
        int first = value & 0x3F;
        long pos = this.address + (long)writerIdx;
        UNSAFE.putByte(this.heapMemory, pos, (byte)(first | 0x80));
        UNSAFE.putByte(this.heapMemory, pos + 1L, (byte)(value >>> 6 | 0x80));
        UNSAFE.putByte(this.heapMemory, pos + 2L, (byte)(value >>> 12 | 0x80));
        if (numPaddingBytes == 4) {
            UNSAFE.putByte(this.heapMemory, pos + 3L, (byte)(value >>> 18 | 0x40));
            this.writerIndex = writerIdx + 4;
            return 4;
        }
        UNSAFE.putByte(this.heapMemory, pos + 3L, (byte)(value >>> 18));
        UNSAFE.putInt(this.heapMemory, pos + 4L, 0);
        UNSAFE.putByte(this.heapMemory, pos + (long)numPaddingBytes + 3L, (byte)64);
        this.writerIndex = writerIdx + numPaddingBytes + 4;
        return numPaddingBytes + 4;
    }

    private int writeVarUint32Aligned5(int value) {
        int writerIdx = this.writerIndex;
        int numPaddingBytes = 4 - writerIdx % 4;
        this.ensure(writerIdx + 9);
        int first = value & 0x3F;
        long pos = this.address + (long)writerIdx;
        UNSAFE.putByte(this.heapMemory, pos, (byte)(first | 0x80));
        UNSAFE.putByte(this.heapMemory, pos + 1L, (byte)(value >>> 6 | 0x80));
        UNSAFE.putByte(this.heapMemory, pos + 2L, (byte)(value >>> 12 | 0x80));
        UNSAFE.putByte(this.heapMemory, pos + 3L, (byte)(value >>> 18 | 0x80));
        if (numPaddingBytes == 1) {
            UNSAFE.putByte(this.heapMemory, pos + 4L, (byte)(value >>> 24 | 0x40));
            this.writerIndex = writerIdx + 5;
            return 5;
        }
        UNSAFE.putByte(this.heapMemory, pos + 4L, (byte)(value >>> 24));
        UNSAFE.putInt(this.heapMemory, pos + 5L, 0);
        UNSAFE.putByte(this.heapMemory, pos + (long)numPaddingBytes + 3L, (byte)64);
        this.writerIndex = writerIdx + numPaddingBytes + 4;
        return numPaddingBytes + 4;
    }

    private int writeVarUint32Aligned6(int value) {
        int writerIdx = this.writerIndex;
        int numPaddingBytes = 4 - writerIdx % 4;
        this.ensure(writerIdx + 10);
        int first = value & 0x3F;
        long pos = this.address + (long)writerIdx;
        UNSAFE.putByte(this.heapMemory, pos, (byte)(first | 0x80));
        UNSAFE.putByte(this.heapMemory, pos + 1L, (byte)(value >>> 6 | 0x80));
        UNSAFE.putByte(this.heapMemory, pos + 2L, (byte)(value >>> 12 | 0x80));
        UNSAFE.putByte(this.heapMemory, pos + 3L, (byte)(value >>> 18 | 0x80));
        UNSAFE.putByte(this.heapMemory, pos + 4L, (byte)(value >>> 24 | 0x80));
        if (numPaddingBytes == 2) {
            UNSAFE.putByte(this.heapMemory, pos + 5L, (byte)(value >>> 30 | 0x40));
            this.writerIndex = writerIdx + 6;
            return 6;
        }
        UNSAFE.putByte(this.heapMemory, pos + 5L, (byte)(value >>> 30));
        UNSAFE.putInt(this.heapMemory, pos + 6L, 0);
        if (numPaddingBytes == 1) {
            UNSAFE.putByte(this.heapMemory, pos + 8L, (byte)64);
            this.writerIndex = writerIdx + 9;
            return 9;
        }
        UNSAFE.putByte(this.heapMemory, pos + (long)numPaddingBytes + 3L, (byte)64);
        this.writerIndex = writerIdx + numPaddingBytes + 4;
        return numPaddingBytes + 4;
    }

    public int writeVarInt64(long value) {
        this.ensure(this.writerIndex + 9);
        return this._unsafeWriteVarUint64(value << 1 ^ value >> 63);
    }

    public int _unsafeWriteVarInt64(long value) {
        return this._unsafeWriteVarUint64(value << 1 ^ value >> 63);
    }

    public int writeVarUint64(long value) {
        this.ensure(this.writerIndex + 9);
        return this._unsafeWriteVarUint64(value);
    }

    public int _unsafeWriteVarUint64(long value) {
        int writerIndex = this.writerIndex;
        int varInt = (int)(value & 0x7FL);
        if (value >>> 7 == 0L) {
            UNSAFE.putByte(this.heapMemory, this.address + (long)writerIndex, (byte)varInt);
            this.writerIndex = writerIndex + 1;
            return 1;
        }
        varInt |= (int)((value & 0x3F80L) << 1 | 0x80L);
        if (value >>> 14 == 0L) {
            this._unsafePutInt32(writerIndex, varInt);
            this.writerIndex = writerIndex + 2;
            return 2;
        }
        varInt |= (int)((value & 0x1FC000L) << 2 | 0x8000L);
        if (value >>> 21 == 0L) {
            this._unsafePutInt32(writerIndex, varInt);
            this.writerIndex = writerIndex + 3;
            return 3;
        }
        varInt |= (int)((value & 0xFE00000L) << 3 | 0x800000L);
        if (value >>> 28 == 0L) {
            this._unsafePutInt32(writerIndex, varInt);
            this.writerIndex = writerIndex + 4;
            return 4;
        }
        long varLong = (long)varInt & 0xFFFFFFFFL;
        varLong |= (value & 0x7F0000000L) << 4 | 0x80000000L;
        if (value >>> 35 == 0L) {
            this._unsafePutInt64(writerIndex, varLong);
            this.writerIndex = writerIndex + 5;
            return 5;
        }
        varLong |= (value & 0x3F800000000L) << 5 | 0x8000000000L;
        if (value >>> 42 == 0L) {
            this._unsafePutInt64(writerIndex, varLong);
            this.writerIndex = writerIndex + 6;
            return 6;
        }
        varLong |= (value & 0x1FC0000000000L) << 6 | 0x800000000000L;
        if (value >>> 49 == 0L) {
            this._unsafePutInt64(writerIndex, varLong);
            this.writerIndex = writerIndex + 7;
            return 7;
        }
        varLong |= (value & 0xFE000000000000L) << 7 | 0x80000000000000L;
        if ((value >>>= 56) == 0L) {
            this._unsafePutInt64(writerIndex, varLong);
            this.writerIndex = writerIndex + 8;
            return 8;
        }
        this._unsafePutInt64(writerIndex, varLong | Long.MIN_VALUE);
        UNSAFE.putByte(this.heapMemory, this.address + (long)writerIndex + 8L, (byte)(value & 0xFFL));
        this.writerIndex = writerIndex + 9;
        return 9;
    }

    public int writeSliInt64(long value) {
        this.ensure(this.writerIndex + 9);
        return this._unsafeWriteSliInt64(value);
    }

    public int _unsafeWriteSliInt64(long value) {
        int writerIndex = this.writerIndex;
        long pos = this.address + (long)writerIndex;
        byte[] heapMemory = this.heapMemory;
        if (value >= -1073741824L && value <= 0x3FFFFFFFL) {
            int v = (int)value << 1;
            if (!LITTLE_ENDIAN) {
                v = Integer.reverseBytes(v);
            }
            UNSAFE.putInt(heapMemory, pos, v);
            this.writerIndex = writerIndex + 4;
            return 4;
        }
        UNSAFE.putByte(heapMemory, pos, (byte)1);
        if (!LITTLE_ENDIAN) {
            value = Long.reverseBytes(value);
        }
        UNSAFE.putLong(heapMemory, pos + 1L, value);
        this.writerIndex = writerIndex + 9;
        return 9;
    }

    public void writeBytes(byte[] bytes) {
        this.writeBytes(bytes, 0, bytes.length);
    }

    public void writeBytes(byte[] bytes, int offset, int length) {
        int writerIdx = this.writerIndex;
        int newIdx = writerIdx + length;
        this.ensure(newIdx);
        this.put(writerIdx, bytes, offset, length);
        this.writerIndex = newIdx;
    }

    public void write(ByteBuffer source) {
        this.write(source, source.remaining());
    }

    public void write(ByteBuffer source, int numBytes) {
        int writerIdx = this.writerIndex;
        int newIdx = writerIdx + numBytes;
        this.ensure(newIdx);
        this.put(writerIdx, source, numBytes);
        this.writerIndex = newIdx;
    }

    public void writePrimitiveArrayWithSize(Object arr, int offset, int numBytes) {
        int idx = this.writerIndex;
        this.ensure(idx + 5 + numBytes);
        Platform.copyMemory(arr, offset, this.heapMemory, this.address + (long)(idx += this._unsafeWriteVarUint32(numBytes)), numBytes);
        this.writerIndex = idx + numBytes;
    }

    public void writePrimitiveArrayAlignedSize(Object arr, int offset, int numBytes) {
        this.writeVarUint32Aligned(numBytes);
        int writerIdx = this.writerIndex;
        int newIdx = writerIdx + numBytes;
        this.ensure(newIdx);
        Platform.copyMemory(arr, offset, this.heapMemory, this.address + (long)writerIdx, numBytes);
        this.writerIndex = newIdx;
    }

    public void writePrimitiveArray(Object arr, int offset, int numBytes) {
        int writerIdx = this.writerIndex;
        int newIdx = writerIdx + numBytes;
        this.ensure(newIdx);
        Platform.copyMemory(arr, offset, this.heapMemory, this.address + (long)writerIdx, numBytes);
        this.writerIndex = newIdx;
    }

    public void grow(int neededSize) {
        int length = this.writerIndex + neededSize;
        if (length > this.size) {
            this.growBuffer(length);
        }
    }

    public void ensure(int length) {
        if (length > this.size) {
            this.growBuffer(length);
        }
    }

    private void growBuffer(int length) {
        int newSize = length < 0x6400000 ? length << 2 : (int)Math.min((double)length * 1.5, 2.147483639E9);
        byte[] data = new byte[newSize];
        this.copyToUnsafe(0L, data, Platform.BYTE_ARRAY_OFFSET, this.size());
        this.initHeapBuffer(data, 0, data.length);
    }

    private void throwIndexOOBExceptionForRead() {
        throw new IndexOutOfBoundsException(String.format("readerIndex: %d (expected: 0 <= readerIndex <= size(%d))", this.readerIndex, this.size));
    }

    private void throwIndexOOBExceptionForRead(int length) {
        throw new IndexOutOfBoundsException(String.format("readerIndex: %d (expected: 0 <= readerIndex <= size(%d)), length %d", this.readerIndex, this.size, length));
    }

    public int readerIndex() {
        return this.readerIndex;
    }

    public void readerIndex(int readerIndex) {
        if (readerIndex < 0) {
            this.throwIndexOOBExceptionForRead();
        } else if (readerIndex > this.size) {
            this.streamReader.fillBuffer(readerIndex - this.size);
        }
        this.readerIndex = readerIndex;
    }

    public int _unsafeHeapReaderIndex() {
        return this.readerIndex + this.heapOffset;
    }

    public void _increaseReaderIndexUnsafe(int diff) {
        this.readerIndex += diff;
    }

    public void increaseReaderIndex(int diff) {
        int readerIdx = this.readerIndex;
        this.readerIndex = readerIdx += diff;
        if (readerIdx < 0) {
            this.throwIndexOOBExceptionForRead();
        } else if (readerIdx > this.size) {
            this.streamReader.fillBuffer(readerIdx - this.size);
        }
    }

    public long getUnsafeReaderAddress() {
        return this.address + (long)this.readerIndex;
    }

    public int remaining() {
        return this.size - this.readerIndex;
    }

    public boolean readBoolean() {
        int readerIdx = this.readerIndex;
        if (readerIdx > this.size - 1) {
            this.streamReader.fillBuffer(1);
        }
        this.readerIndex = readerIdx + 1;
        return UNSAFE.getByte(this.heapMemory, this.address + (long)readerIdx) != 0;
    }

    public int readUnsignedByte() {
        int readerIdx = this.readerIndex;
        if (readerIdx > this.size - 1) {
            this.streamReader.fillBuffer(1);
        }
        this.readerIndex = readerIdx + 1;
        int v = UNSAFE.getByte(this.heapMemory, this.address + (long)readerIdx);
        return v &= 0xFF;
    }

    public byte readByte() {
        int readerIdx = this.readerIndex;
        if (readerIdx > this.size - 1) {
            this.streamReader.fillBuffer(1);
        }
        this.readerIndex = readerIdx + 1;
        return UNSAFE.getByte(this.heapMemory, this.address + (long)readerIdx);
    }

    public char readChar() {
        int readerIdx = this.readerIndex;
        int remaining = this.size - readerIdx;
        if (remaining < 2) {
            this.streamReader.fillBuffer(2 - remaining);
        }
        this.readerIndex = readerIdx + 2;
        char c = UNSAFE.getChar(this.heapMemory, this.address + (long)readerIdx);
        return LITTLE_ENDIAN ? c : Character.reverseBytes(c);
    }

    public short readInt16() {
        int readerIdx = this.readerIndex;
        int remaining = this.size - readerIdx;
        if (remaining < 2) {
            this.streamReader.fillBuffer(2 - remaining);
        }
        this.readerIndex = readerIdx + 2;
        short v = UNSAFE.getShort(this.heapMemory, this.address + (long)readerIdx);
        return LITTLE_ENDIAN ? v : Short.reverseBytes(v);
    }

    public short _readInt16OnLE() {
        int readerIdx = this.readerIndex;
        int remaining = this.size - readerIdx;
        if (remaining < 2) {
            this.streamReader.fillBuffer(2 - remaining);
        }
        this.readerIndex = readerIdx + 2;
        return UNSAFE.getShort(this.heapMemory, this.address + (long)readerIdx);
    }

    public short _readInt16OnBE() {
        int readerIdx = this.readerIndex;
        int remaining = this.size - readerIdx;
        if (remaining < 2) {
            this.streamReader.fillBuffer(2 - remaining);
        }
        this.readerIndex = readerIdx + 2;
        return Short.reverseBytes(UNSAFE.getShort(this.heapMemory, this.address + (long)readerIdx));
    }

    public int readInt32() {
        int readerIdx = this.readerIndex;
        int remaining = this.size - readerIdx;
        if (remaining < 4) {
            this.streamReader.fillBuffer(4 - remaining);
        }
        this.readerIndex = readerIdx + 4;
        int v = UNSAFE.getInt(this.heapMemory, this.address + (long)readerIdx);
        return LITTLE_ENDIAN ? v : Integer.reverseBytes(v);
    }

    public int _readInt32OnLE() {
        int readerIdx = this.readerIndex;
        int remaining = this.size - readerIdx;
        if (remaining < 4) {
            this.streamReader.fillBuffer(4 - remaining);
        }
        this.readerIndex = readerIdx + 4;
        return UNSAFE.getInt(this.heapMemory, this.address + (long)readerIdx);
    }

    public int _readInt32OnBE() {
        int readerIdx = this.readerIndex;
        int remaining = this.size - readerIdx;
        if (remaining < 4) {
            this.streamReader.fillBuffer(4 - remaining);
        }
        this.readerIndex = readerIdx + 4;
        return Integer.reverseBytes(UNSAFE.getInt(this.heapMemory, this.address + (long)readerIdx));
    }

    public long readInt64() {
        int readerIdx = this.readerIndex;
        int remaining = this.size - readerIdx;
        if (remaining < 8) {
            this.streamReader.fillBuffer(8 - remaining);
        }
        this.readerIndex = readerIdx + 8;
        long v = UNSAFE.getLong(this.heapMemory, this.address + (long)readerIdx);
        return LITTLE_ENDIAN ? v : Long.reverseBytes(v);
    }

    public long _readInt64OnLE() {
        int readerIdx = this.readerIndex;
        int remaining = this.size - readerIdx;
        if (remaining < 8) {
            this.streamReader.fillBuffer(8 - remaining);
        }
        this.readerIndex = readerIdx + 8;
        return UNSAFE.getLong(this.heapMemory, this.address + (long)readerIdx);
    }

    public long _readInt64OnBE() {
        int readerIdx = this.readerIndex;
        int remaining = this.size - readerIdx;
        if (remaining < 8) {
            this.streamReader.fillBuffer(8 - remaining);
        }
        this.readerIndex = readerIdx + 8;
        return Long.reverseBytes(UNSAFE.getLong(this.heapMemory, this.address + (long)readerIdx));
    }

    public long readSliInt64() {
        if (LITTLE_ENDIAN) {
            return this._readSliInt64OnLE();
        }
        return this._readSliInt64OnBE();
    }

    public long _readSliInt64OnLE() {
        int i;
        int readIdx = this.readerIndex;
        int diff = this.size - readIdx;
        if (diff < 4) {
            this.streamReader.fillBuffer(4 - diff);
        }
        if (((i = UNSAFE.getInt(this.heapMemory, this.address + (long)readIdx)) & 1) != 1) {
            this.readerIndex = readIdx + 4;
            return i >> 1;
        }
        diff = this.size - readIdx;
        if (diff < 9) {
            this.streamReader.fillBuffer(9 - diff);
        }
        this.readerIndex = readIdx + 9;
        return UNSAFE.getLong(this.heapMemory, this.address + (long)readIdx + 1L);
    }

    public long _readSliInt64OnBE() {
        int i;
        int readIdx = this.readerIndex;
        int diff = this.size - readIdx;
        if (diff < 4) {
            this.streamReader.fillBuffer(4 - diff);
        }
        if (((i = Integer.reverseBytes(UNSAFE.getInt(this.heapMemory, this.address + (long)readIdx))) & 1) != 1) {
            this.readerIndex = readIdx + 4;
            return i >> 1;
        }
        diff = this.size - readIdx;
        if (diff < 9) {
            this.streamReader.fillBuffer(9 - diff);
        }
        this.readerIndex = readIdx + 9;
        return Long.reverseBytes(UNSAFE.getLong(this.heapMemory, this.address + (long)readIdx + 1L));
    }

    public float readFloat32() {
        int readerIdx = this.readerIndex;
        int remaining = this.size - readerIdx;
        if (remaining < 4) {
            this.streamReader.fillBuffer(4 - remaining);
        }
        this.readerIndex = readerIdx + 4;
        int v = UNSAFE.getInt(this.heapMemory, this.address + (long)readerIdx);
        if (!LITTLE_ENDIAN) {
            v = Integer.reverseBytes(v);
        }
        return Float.intBitsToFloat(v);
    }

    public float _readFloat32OnLE() {
        int readerIdx = this.readerIndex;
        int remaining = this.size - readerIdx;
        if (remaining < 4) {
            this.streamReader.fillBuffer(4 - remaining);
        }
        this.readerIndex = readerIdx + 4;
        return Float.intBitsToFloat(UNSAFE.getInt(this.heapMemory, this.address + (long)readerIdx));
    }

    public float _readFloat32OnBE() {
        int readerIdx = this.readerIndex;
        int remaining = this.size - readerIdx;
        if (remaining < 4) {
            this.streamReader.fillBuffer(4 - remaining);
        }
        this.readerIndex = readerIdx + 4;
        return Float.intBitsToFloat(Integer.reverseBytes(UNSAFE.getInt(this.heapMemory, this.address + (long)readerIdx)));
    }

    public double readFloat64() {
        int readerIdx = this.readerIndex;
        int remaining = this.size - readerIdx;
        if (remaining < 8) {
            this.streamReader.fillBuffer(8 - remaining);
        }
        this.readerIndex = readerIdx + 8;
        long v = UNSAFE.getLong(this.heapMemory, this.address + (long)readerIdx);
        if (!LITTLE_ENDIAN) {
            v = Long.reverseBytes(v);
        }
        return Double.longBitsToDouble(v);
    }

    public double _readFloat64OnLE() {
        int readerIdx = this.readerIndex;
        int remaining = this.size - readerIdx;
        if (remaining < 8) {
            this.streamReader.fillBuffer(8 - remaining);
        }
        this.readerIndex = readerIdx + 8;
        return Double.longBitsToDouble(UNSAFE.getLong(this.heapMemory, this.address + (long)readerIdx));
    }

    public double _readFloat64OnBE() {
        int readerIdx = this.readerIndex;
        int remaining = this.size - readerIdx;
        if (remaining < 8) {
            this.streamReader.fillBuffer(8 - remaining);
        }
        this.readerIndex = readerIdx + 8;
        return Double.longBitsToDouble(Long.reverseBytes(UNSAFE.getLong(this.heapMemory, this.address + (long)readerIdx)));
    }

    public int readVarInt32() {
        if (LITTLE_ENDIAN) {
            return this._readVarInt32OnLE();
        }
        return this._readVarInt32OnBE();
    }

    public int _readVarInt32OnLE() {
        int result;
        int readIdx = this.readerIndex;
        if (this.size - readIdx < 5) {
            result = (int)this.readVarUint36Slow();
        } else {
            long address = this.address;
            int fourByteValue = UNSAFE.getInt(this.heapMemory, address + (long)readIdx);
            ++readIdx;
            result = fourByteValue & 0x7F;
            if ((fourByteValue & 0x80) != 0) {
                ++readIdx;
                result |= fourByteValue >>> 1 & 0x3F80;
                if ((fourByteValue & 0x8000) != 0) {
                    ++readIdx;
                    result |= fourByteValue >>> 2 & 0x1FC000;
                    if ((fourByteValue & 0x800000) != 0) {
                        ++readIdx;
                        result |= fourByteValue >>> 3 & 0xFE00000;
                        if ((fourByteValue & Integer.MIN_VALUE) != 0) {
                            result |= (UNSAFE.getByte(this.heapMemory, address + (long)readIdx++) & 0x7F) << 28;
                        }
                    }
                }
            }
            this.readerIndex = readIdx;
        }
        return result >>> 1 ^ -(result & 1);
    }

    public int _readVarInt32OnBE() {
        int result;
        int readIdx = this.readerIndex;
        if (this.size - readIdx < 5) {
            result = (int)this.readVarUint36Slow();
        } else {
            long address = this.address;
            int fourByteValue = Integer.reverseBytes(UNSAFE.getInt(this.heapMemory, address + (long)readIdx));
            ++readIdx;
            result = fourByteValue & 0x7F;
            if ((fourByteValue & 0x80) != 0) {
                ++readIdx;
                result |= fourByteValue >>> 1 & 0x3F80;
                if ((fourByteValue & 0x8000) != 0) {
                    ++readIdx;
                    result |= fourByteValue >>> 2 & 0x1FC000;
                    if ((fourByteValue & 0x800000) != 0) {
                        ++readIdx;
                        result |= fourByteValue >>> 3 & 0xFE00000;
                        if ((fourByteValue & Integer.MIN_VALUE) != 0) {
                            result |= (UNSAFE.getByte(this.heapMemory, address + (long)readIdx++) & 0x7F) << 28;
                        }
                    }
                }
            }
            this.readerIndex = readIdx;
        }
        return result >>> 1 ^ -(result & 1);
    }

    public long readVarUint36Small() {
        int readIdx = this.readerIndex;
        if (this.size - readIdx >= 9) {
            long bulkValue = this._unsafeGetInt64(readIdx++);
            long result = bulkValue & 0x7FL;
            if ((bulkValue & 0x80L) != 0L) {
                ++readIdx;
                result |= bulkValue >>> 1 & 0x3F80L;
                if ((bulkValue & 0x8000L) != 0L) {
                    return this.continueReadVarInt36(readIdx, bulkValue, result);
                }
            }
            this.readerIndex = readIdx;
            return result;
        }
        return this.readVarUint36Slow();
    }

    private long continueReadVarInt36(int readIdx, long bulkValue, long result) {
        ++readIdx;
        result |= bulkValue >>> 2 & 0x1FC000L;
        if ((bulkValue & 0x800000L) != 0L) {
            ++readIdx;
            result |= bulkValue >>> 3 & 0xFE00000L;
            if ((bulkValue & 0x80000000L) != 0L) {
                ++readIdx;
                result |= bulkValue >>> 4 & 0xFF0000000L;
            }
        }
        this.readerIndex = readIdx;
        return result;
    }

    private long readVarUint36Slow() {
        long b = this.readByte();
        long result = b & 0x7FL;
        if ((b & 0x80L) != 0L) {
            b = this.readByte();
            result |= (b & 0x7FL) << 7;
            if ((b & 0x80L) != 0L) {
                b = this.readByte();
                result |= (b & 0x7FL) << 14;
                if ((b & 0x80L) != 0L) {
                    b = this.readByte();
                    result |= (b & 0x7FL) << 21;
                    if ((b & 0x80L) != 0L) {
                        b = this.readByte();
                        result |= (b & 0xFFL) << 28;
                    }
                }
            }
        }
        return result;
    }

    public int readVarUint32() {
        int readIdx = this.readerIndex;
        if (this.size - readIdx < 5) {
            return (int)this.readVarUint36Slow();
        }
        int fourByteValue = this._unsafeGetInt32(readIdx);
        ++readIdx;
        int result = fourByteValue & 0x7F;
        if ((fourByteValue & 0x80) != 0) {
            ++readIdx;
            result |= fourByteValue >>> 1 & 0x3F80;
            if ((fourByteValue & 0x8000) != 0) {
                ++readIdx;
                result |= fourByteValue >>> 2 & 0x1FC000;
                if ((fourByteValue & 0x800000) != 0) {
                    ++readIdx;
                    result |= fourByteValue >>> 3 & 0xFE00000;
                    if ((fourByteValue & Integer.MIN_VALUE) != 0) {
                        result |= (UNSAFE.getByte(this.heapMemory, this.address + (long)readIdx++) & 0x7F) << 28;
                    }
                }
            }
        }
        this.readerIndex = readIdx;
        return result;
    }

    public int readVarUint32Small7() {
        byte v;
        int readIdx = this.readerIndex;
        if (this.size - readIdx > 0 && ((v = UNSAFE.getByte(this.heapMemory, this.address + (long)readIdx++)) & 0x80) == 0) {
            this.readerIndex = readIdx;
            return v;
        }
        return this.readVarUint32Small14();
    }

    public int readVarUint32Small14() {
        int readIdx = this.readerIndex;
        if (this.size - readIdx >= 5) {
            int fourByteValue = this._unsafeGetInt32(readIdx++);
            int value = fourByteValue & 0x7F;
            if ((fourByteValue & 0x80) != 0) {
                ++readIdx;
                value |= fourByteValue >>> 1 & 0x3F80;
                if ((fourByteValue & 0x8000) != 0) {
                    return this.continueReadVarUint32(readIdx, fourByteValue, value);
                }
            }
            this.readerIndex = readIdx;
            return value;
        }
        return (int)this.readVarUint36Slow();
    }

    private int continueReadVarUint32(int readIdx, int bulkRead, int value) {
        ++readIdx;
        value |= bulkRead >>> 2 & 0x1FC000;
        if ((bulkRead & 0x800000) != 0) {
            ++readIdx;
            value |= bulkRead >>> 3 & 0xFE00000;
            if ((bulkRead & Integer.MIN_VALUE) != 0) {
                value |= (UNSAFE.getByte(this.heapMemory, this.address + (long)readIdx++) & 0x7F) << 28;
            }
        }
        this.readerIndex = readIdx;
        return value;
    }

    public long readVarInt64() {
        return LITTLE_ENDIAN ? this._readVarInt64OnLE() : this._readVarInt64OnBE();
    }

    public long _readVarInt64OnLE() {
        long result;
        int readIdx = this.readerIndex;
        if (this.size - readIdx < 9) {
            result = this.readVarUint64Slow();
        } else {
            long address = this.address;
            long bulkValue = UNSAFE.getLong(this.heapMemory, address + (long)readIdx);
            ++readIdx;
            result = bulkValue & 0x7FL;
            if ((bulkValue & 0x80L) != 0L) {
                ++readIdx;
                result |= bulkValue >>> 1 & 0x3F80L;
                if ((bulkValue & 0x8000L) != 0L) {
                    result = this.continueReadVarInt64(readIdx, bulkValue, result);
                    return result >>> 1 ^ -(result & 1L);
                }
            }
            this.readerIndex = readIdx;
        }
        return result >>> 1 ^ -(result & 1L);
    }

    public long _readVarInt64OnBE() {
        long result;
        int readIdx = this.readerIndex;
        if (this.size - readIdx < 9) {
            result = this.readVarUint64Slow();
        } else {
            long address = this.address;
            long bulkValue = Long.reverseBytes(UNSAFE.getLong(this.heapMemory, address + (long)readIdx));
            ++readIdx;
            result = bulkValue & 0x7FL;
            if ((bulkValue & 0x80L) != 0L) {
                ++readIdx;
                result |= bulkValue >>> 1 & 0x3F80L;
                if ((bulkValue & 0x8000L) != 0L) {
                    result = this.continueReadVarInt64(readIdx, bulkValue, result);
                    return result >>> 1 ^ -(result & 1L);
                }
            }
            this.readerIndex = readIdx;
        }
        return result >>> 1 ^ -(result & 1L);
    }

    public long readVarUint64() {
        int readIdx = this.readerIndex;
        if (this.size - readIdx < 9) {
            return this.readVarUint64Slow();
        }
        long bulkValue = this._unsafeGetInt64(readIdx);
        ++readIdx;
        long result = bulkValue & 0x7FL;
        if ((bulkValue & 0x80L) != 0L) {
            ++readIdx;
            result |= bulkValue >>> 1 & 0x3F80L;
            if ((bulkValue & 0x8000L) != 0L) {
                return this.continueReadVarInt64(readIdx, bulkValue, result);
            }
        }
        this.readerIndex = readIdx;
        return result;
    }

    private long continueReadVarInt64(int readIdx, long bulkValue, long result) {
        ++readIdx;
        result |= bulkValue >>> 2 & 0x1FC000L;
        if ((bulkValue & 0x800000L) != 0L) {
            ++readIdx;
            result |= bulkValue >>> 3 & 0xFE00000L;
            if ((bulkValue & 0x80000000L) != 0L) {
                ++readIdx;
                result |= bulkValue >>> 4 & 0x7F0000000L;
                if ((bulkValue & 0x8000000000L) != 0L) {
                    ++readIdx;
                    result |= bulkValue >>> 5 & 0x3F800000000L;
                    if ((bulkValue & 0x800000000000L) != 0L) {
                        ++readIdx;
                        result |= bulkValue >>> 6 & 0x1FC0000000000L;
                        if ((bulkValue & 0x80000000000000L) != 0L) {
                            ++readIdx;
                            result |= bulkValue >>> 7 & 0xFE000000000000L;
                            if ((bulkValue & Long.MIN_VALUE) != 0L) {
                                long b = UNSAFE.getByte(this.heapMemory, this.address + (long)readIdx++);
                                result |= b << 56;
                            }
                        }
                    }
                }
            }
        }
        this.readerIndex = readIdx;
        return result;
    }

    private long readVarUint64Slow() {
        long b = this.readByte();
        long result = b & 0x7FL;
        if ((b & 0x80L) != 0L) {
            b = this.readByte();
            result |= (b & 0x7FL) << 7;
            if ((b & 0x80L) != 0L) {
                b = this.readByte();
                result |= (b & 0x7FL) << 14;
                if ((b & 0x80L) != 0L) {
                    b = this.readByte();
                    result |= (b & 0x7FL) << 21;
                    if ((b & 0x80L) != 0L) {
                        b = this.readByte();
                        result |= (b & 0x7FL) << 28;
                        if ((b & 0x80L) != 0L) {
                            b = this.readByte();
                            result |= (b & 0x7FL) << 35;
                            if ((b & 0x80L) != 0L) {
                                b = this.readByte();
                                result |= (b & 0x7FL) << 42;
                                if ((b & 0x80L) != 0L) {
                                    b = this.readByte();
                                    result |= (b & 0x7FL) << 49;
                                    if ((b & 0x80L) != 0L) {
                                        b = this.readByte();
                                        result |= b << 56;
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
        return result;
    }

    public int readAlignedVarUint() {
        long pos;
        int readerIdx = this.readerIndex;
        if (readerIdx < this.size - 10) {
            return this.slowReadAlignedVarUint();
        }
        long startPos = pos = this.address + (long)readerIdx;
        byte b = UNSAFE.getByte(this.heapMemory, pos++);
        int result = b & 0x3F;
        if ((b & 0x80) != 0) {
            b = UNSAFE.getByte(this.heapMemory, pos++);
            result |= (b & 0x3F) << 6;
            if ((b & 0x80) != 0) {
                b = UNSAFE.getByte(this.heapMemory, pos++);
                result |= (b & 0x3F) << 12;
                if ((b & 0x80) != 0) {
                    b = UNSAFE.getByte(this.heapMemory, pos++);
                    result |= (b & 0x3F) << 18;
                    if ((b & 0x80) != 0) {
                        b = UNSAFE.getByte(this.heapMemory, pos++);
                        result |= (b & 0x3F) << 24;
                        if ((b & 0x80) != 0) {
                            b = UNSAFE.getByte(this.heapMemory, pos++);
                            result |= (b & 0x3F) << 30;
                        }
                    }
                }
            }
        }
        pos = this.skipPadding(pos, b);
        this.readerIndex = (int)(pos - startPos + (long)readerIdx);
        return result;
    }

    public int slowReadAlignedVarUint() {
        byte b = this.readByte();
        int result = b & 0x3F;
        if ((b & 0x80) != 0) {
            b = this.readByte();
            result |= (b & 0x3F) << 6;
            if ((b & 0x80) != 0) {
                b = this.readByte();
                result |= (b & 0x3F) << 12;
                if ((b & 0x80) != 0) {
                    b = this.readByte();
                    result |= (b & 0x3F) << 18;
                    if ((b & 0x80) != 0) {
                        b = this.readByte();
                        result |= (b & 0x3F) << 24;
                        if ((b & 0x80) != 0) {
                            b = this.readByte();
                            result |= (b & 0x3F) << 30;
                        }
                    }
                }
            }
        }
        if ((b & 0x40) == 0 && ((b = this.readByte()) & 0x40) == 0 && ((b = this.readByte()) & 0x40) == 0) {
            b = this.readByte();
            Preconditions.checkArgument((b & 0x40) != 0, "At most 3 padding bytes.");
        }
        return result;
    }

    private long skipPadding(long pos, int b) {
        if ((b & 0x40) == 0 && ((b = (int)UNSAFE.getByte(this.heapMemory, pos++)) & 0x40) == 0 && ((b = (int)UNSAFE.getByte(this.heapMemory, pos++)) & 0x40) == 0) {
            Preconditions.checkArgument(((b = (int)UNSAFE.getByte(this.heapMemory, pos++)) & 0x40) != 0, "At most 3 padding bytes.");
        }
        return pos;
    }

    public byte[] readBytes(int length) {
        int readerIdx = this.readerIndex;
        byte[] bytes = new byte[length];
        if (length > this.size - readerIdx) {
            this.streamReader.readTo(bytes, 0, length);
            return bytes;
        }
        byte[] heapMemory = this.heapMemory;
        if (heapMemory != null) {
            System.arraycopy(heapMemory, this.heapOffset + readerIdx, bytes, 0, length);
        } else {
            Platform.copyMemory(null, this.address + (long)readerIdx, bytes, Platform.BYTE_ARRAY_OFFSET, length);
        }
        this.readerIndex = readerIdx + length;
        return bytes;
    }

    public void readBytes(byte[] dst, int dstIndex, int length) {
        int readerIdx = this.readerIndex;
        if (readerIdx > this.size - length) {
            this.streamReader.readTo(dst, dstIndex, length);
            return;
        }
        if (dstIndex < 0 || dstIndex > dst.length - length) {
            this.throwIndexOOBExceptionForRead();
        }
        this.copyToUnsafe(readerIdx, dst, Platform.BYTE_ARRAY_OFFSET + dstIndex, length);
        this.readerIndex = readerIdx + length;
    }

    public void readBytes(byte[] dst) {
        this.readBytes(dst, 0, dst.length);
    }

    public long readBytesAsInt64(int len) {
        int readerIdx = this.readerIndex;
        int remaining = this.size - readerIdx;
        if (remaining >= 8) {
            this.readerIndex = readerIdx + len;
            long v = UNSAFE.getLong(this.heapMemory, this.address + (long)readerIdx);
            v = (LITTLE_ENDIAN ? v : Long.reverseBytes(v)) & -1L >>> (8 - len) * 8;
            return v;
        }
        return this.slowReadBytesAsInt64(remaining, len);
    }

    private long slowReadBytesAsInt64(int remaining, int len) {
        if (remaining < len) {
            this.streamReader.fillBuffer(len - remaining);
        }
        int readerIdx = this.readerIndex;
        this.readerIndex = readerIdx + len;
        long result = 0L;
        byte[] heapMemory = this.heapMemory;
        if (heapMemory != null) {
            int start = this.heapOffset + readerIdx;
            for (int i = 0; i < len; ++i) {
                result |= ((long)heapMemory[start + i] & 0xFFL) << i * 8;
            }
        } else {
            long start = this.address + (long)readerIdx;
            for (int i = 0; i < len; ++i) {
                result |= ((long)UNSAFE.getByte(null, start + (long)i) & 0xFFL) << i * 8;
            }
        }
        return result;
    }

    public int read(ByteBuffer dst) {
        int readerIdx = this.readerIndex;
        int len = dst.remaining();
        if (readerIdx > this.size - len) {
            return this.streamReader.readToByteBuffer(dst);
        }
        if (this.heapMemory != null) {
            dst.put(this.heapMemory, this.readerIndex + this.heapOffset, len);
        } else {
            dst.put(this.sliceAsByteBuffer(readerIdx, len));
        }
        this.readerIndex = readerIdx + len;
        return len;
    }

    public void read(ByteBuffer dst, int len) {
        int readerIdx = this.readerIndex;
        if (readerIdx > this.size - len) {
            this.streamReader.readToByteBuffer(dst, len);
        } else {
            if (this.heapMemory != null) {
                dst.put(this.heapMemory, this.readerIndex + this.heapOffset, len);
            } else {
                dst.put(this.sliceAsByteBuffer(readerIdx, len));
            }
            this.readerIndex = readerIdx + len;
        }
    }

    public int readBinarySize() {
        int binarySize;
        int readIdx = this.readerIndex;
        if (this.size - readIdx >= 5) {
            int fourByteValue = this._unsafeGetInt32(readIdx++);
            binarySize = fourByteValue & 0x7F;
            if ((fourByteValue & 0x80) != 0) {
                ++readIdx;
                binarySize |= fourByteValue >>> 1 & 0x3F80;
                if ((fourByteValue & 0x8000) != 0) {
                    return this.continueReadBinarySize(readIdx, fourByteValue, binarySize);
                }
            }
            this.readerIndex = readIdx;
        } else {
            binarySize = (int)this.readVarUint36Slow();
            readIdx = this.readerIndex;
        }
        int diff = this.size - readIdx;
        if (diff < binarySize) {
            this.streamReader.fillBuffer(diff);
        }
        return binarySize;
    }

    private int continueReadBinarySize(int readIdx, int bulkRead, int binarySize) {
        int diff;
        ++readIdx;
        binarySize |= bulkRead >>> 2 & 0x1FC000;
        if ((bulkRead & 0x800000) != 0) {
            ++readIdx;
            binarySize |= bulkRead >>> 3 & 0xFE00000;
            if ((bulkRead & Integer.MIN_VALUE) != 0) {
                binarySize |= (UNSAFE.getByte(this.heapMemory, this.address + (long)readIdx++) & 0x7F) << 28;
            }
        }
        if ((diff = this.size - readIdx) < binarySize) {
            this.streamReader.fillBuffer(diff);
        }
        return binarySize;
    }

    public byte[] readBytesAndSize() {
        int numBytes = this.readBinarySize();
        int readerIdx = this.readerIndex;
        byte[] arr = new byte[numBytes];
        if (readerIdx > this.size - numBytes) {
            this.streamReader.readTo(arr, 0, numBytes);
            return arr;
        }
        byte[] heapMemory = this.heapMemory;
        if (heapMemory != null) {
            System.arraycopy(heapMemory, this.heapOffset + readerIdx, arr, 0, numBytes);
        } else {
            Platform.UNSAFE.copyMemory(null, this.address + (long)readerIdx, arr, Platform.BYTE_ARRAY_OFFSET, numBytes);
        }
        this.readerIndex = readerIdx + numBytes;
        return arr;
    }

    public byte[] readBytesWithAlignedSize() {
        int numBytes = this.readAlignedVarUint();
        int readerIdx = this.readerIndex;
        byte[] arr = new byte[numBytes];
        if (readerIdx > this.size - numBytes) {
            this.streamReader.readTo(arr, 0, numBytes);
            return arr;
        }
        Platform.UNSAFE.copyMemory(this.heapMemory, this.address + (long)readerIdx, arr, Platform.BYTE_ARRAY_OFFSET, numBytes);
        this.readerIndex = readerIdx + numBytes;
        return arr;
    }

    public char[] readChars(int numBytes) {
        int readerIdx = this.readerIndex;
        char[] chars = new char[numBytes >> 1];
        if (readerIdx > this.size - numBytes) {
            this.streamReader.readToUnsafe(chars, 0L, numBytes);
            return chars;
        }
        Platform.copyMemory(this.heapMemory, this.address + (long)readerIdx, chars, Platform.CHAR_ARRAY_OFFSET, numBytes);
        this.readerIndex = readerIdx + numBytes;
        return chars;
    }

    public void readChars(char[] chars, int offset, int numBytes) {
        int readerIdx = this.readerIndex;
        if (readerIdx > this.size - numBytes) {
            this.streamReader.readToUnsafe(chars, offset, numBytes);
            return;
        }
        Platform.copyMemory(this.heapMemory, this.address + (long)readerIdx, chars, offset, numBytes);
        this.readerIndex = readerIdx + numBytes;
    }

    public char[] readCharsAndSize() {
        int numBytes = this.readBinarySize();
        int readerIdx = this.readerIndex;
        char[] arr = new char[numBytes >> 1];
        if (readerIdx > this.size - numBytes) {
            this.streamReader.readToUnsafe(arr, 0L, numBytes);
            return arr;
        }
        Platform.UNSAFE.copyMemory(this.heapMemory, this.address + (long)readerIdx, arr, Platform.CHAR_ARRAY_OFFSET, numBytes);
        this.readerIndex = readerIdx + numBytes;
        return arr;
    }

    public char[] readCharsWithAlignedSize() {
        int numBytes = this.readAlignedVarUint();
        return this.readChars(numBytes);
    }

    public long[] readLongs(int numBytes) {
        int readerIdx = this.readerIndex;
        int numElements = numBytes >> 3;
        long[] longs = new long[numElements];
        if (readerIdx > this.size - numBytes) {
            this.streamReader.readToUnsafe(longs, 0L, numElements);
            return longs;
        }
        Platform.copyMemory(this.heapMemory, this.address + (long)readerIdx, longs, Platform.LONG_ARRAY_OFFSET, numBytes);
        this.readerIndex = readerIdx + numBytes;
        return longs;
    }

    public void readToUnsafe(Object target, long targetPointer, int numBytes) {
        int remaining = this.size - this.readerIndex;
        if (numBytes > remaining) {
            this.streamReader.readToUnsafe(target, targetPointer, numBytes);
        } else {
            int readerIdx = this.readerIndex;
            Platform.copyMemory(this.heapMemory, this.address + (long)readerIdx, target, targetPointer, numBytes);
            this.readerIndex = readerIdx + numBytes;
        }
    }

    public void checkReadableBytes(int minimumReadableBytes) {
        int remaining = this.size - this.readerIndex;
        if (minimumReadableBytes > remaining) {
            this.streamReader.fillBuffer(minimumReadableBytes - remaining);
        }
    }

    public byte[] getRemainingBytes() {
        int length = this.size - this.readerIndex;
        if (this.heapMemory != null && this.size == length && this.heapOffset == 0) {
            return this.heapMemory;
        }
        return this.getBytes(this.readerIndex, length);
    }

    public void copyToUnsafe(long offset, Object target, long targetPointer, int numBytes) {
        long thisPointer = this.address + offset;
        Preconditions.checkArgument(thisPointer + (long)numBytes <= this.addressLimit);
        Platform.copyMemory(this.heapMemory, thisPointer, target, targetPointer, numBytes);
    }

    public void copyFromUnsafe(long offset, Object source, long sourcePointer, long numBytes) {
        long thisPointer = this.address + offset;
        Preconditions.checkArgument(thisPointer + numBytes <= this.addressLimit);
        Platform.copyMemory(source, sourcePointer, this.heapMemory, thisPointer, numBytes);
    }

    public void copyTo(int offset, MemoryBuffer target, int targetOffset, int numBytes) {
        byte[] thisHeapRef = this.heapMemory;
        byte[] otherHeapRef = target.heapMemory;
        long thisPointer = this.address + (long)offset;
        long otherPointer = target.address + (long)targetOffset;
        if ((numBytes | offset | targetOffset) < 0 || thisPointer > this.addressLimit - (long)numBytes || otherPointer > target.addressLimit - (long)numBytes) {
            throw new IndexOutOfBoundsException(String.format("offset=%d, targetOffset=%d, numBytes=%d, address=%d, targetAddress=%d", offset, targetOffset, numBytes, this.address, target.address));
        }
        UNSAFE.copyMemory(thisHeapRef, thisPointer, otherHeapRef, otherPointer, numBytes);
    }

    public void copyFrom(int offset, MemoryBuffer source, int sourcePointer, int numBytes) {
        source.copyTo(sourcePointer, this, offset, numBytes);
    }

    public byte[] getBytes(int index, int length) {
        if (index == 0 && this.heapMemory != null && this.heapOffset == 0) {
            return Arrays.copyOf(this.heapMemory, length);
        }
        if (index + length > this.size) {
            this.throwIndexOOBExceptionForRead(length);
        }
        byte[] data = new byte[length];
        this.copyToUnsafe(index, data, Platform.BYTE_ARRAY_OFFSET, length);
        return data;
    }

    public void getBytes(int index, byte[] dst, int dstIndex, int length) {
        if (dstIndex > dst.length - length) {
            this.throwOOBException();
        }
        if (index > this.size - length) {
            this.throwOOBException();
        }
        this.copyToUnsafe(index, dst, Platform.BYTE_ARRAY_OFFSET + dstIndex, length);
    }

    public MemoryBuffer slice(int offset) {
        return this.slice(offset, this.size - offset);
    }

    public MemoryBuffer slice(int offset, int length) {
        if (offset + length > this.size) {
            this.throwOOBExceptionForRange(offset, length);
        }
        if (this.heapMemory != null) {
            return new MemoryBuffer(this.heapMemory, this.heapOffset + offset, length);
        }
        return new MemoryBuffer(this.address + (long)offset, length, this.offHeapBuffer);
    }

    public ByteBuffer sliceAsByteBuffer() {
        return this.sliceAsByteBuffer(this.readerIndex, this.size - this.readerIndex);
    }

    public ByteBuffer sliceAsByteBuffer(int offset, int length) {
        if (offset + length > this.size) {
            this.throwOOBExceptionForRange(offset, length);
        }
        if (this.heapMemory != null) {
            return ByteBuffer.wrap(this.heapMemory, this.heapOffset + offset, length).slice();
        }
        ByteBuffer offHeapBuffer = this.offHeapBuffer;
        if (offHeapBuffer != null) {
            ByteBuffer duplicate = offHeapBuffer.duplicate();
            int start = (int)(this.address - ByteBufferUtil.getAddress(duplicate));
            ByteBufferUtil.position(duplicate, start + offset);
            duplicate.limit(start + offset + length);
            return duplicate.slice();
        }
        return ByteBufferUtil.createDirectByteBufferFromNativeAddress(this.address + (long)offset, length);
    }

    private void throwOOBExceptionForRange(int offset, int length) {
        throw new IndexOutOfBoundsException(String.format("offset(%d) + length(%d) exceeds size(%d): %s", offset, length, this.size, this));
    }

    public ForyStreamReader getStreamReader() {
        return this.streamReader;
    }

    public boolean equalTo(MemoryBuffer buf2, int offset1, int offset2, int len) {
        long pos1 = this.address + (long)offset1;
        long pos2 = buf2.address + (long)offset2;
        Preconditions.checkArgument(pos1 < this.addressLimit);
        Preconditions.checkArgument(pos2 < buf2.addressLimit);
        return Platform.arrayEquals(this.heapMemory, pos1, buf2.heapMemory, pos2, len);
    }

    public String toString() {
        return "MemoryBuffer{size=" + this.size + ", readerIndex=" + this.readerIndex + ", writerIndex=" + this.writerIndex + ", heapMemory=" + (this.heapMemory == null ? null : "len(" + this.heapMemory.length + ")") + ", heapOffset=" + this.heapOffset + ", offHeapBuffer=" + this.offHeapBuffer + ", address=" + this.address + ", addressLimit=" + this.addressLimit + '}';
    }

    public void pointTo(byte[] buffer, int offset, int length) {
        this.initHeapBuffer(buffer, offset, length);
    }

    public static MemoryBuffer fromByteArray(byte[] buffer, int offset, int length) {
        return new MemoryBuffer(buffer, offset, length, null);
    }

    public static MemoryBuffer fromByteArray(byte[] buffer, int offset, int length, ForyStreamReader streamReader) {
        return new MemoryBuffer(buffer, offset, length, streamReader);
    }

    public static MemoryBuffer fromByteArray(byte[] buffer) {
        return new MemoryBuffer(buffer, 0, buffer.length);
    }

    public static MemoryBuffer fromByteBuffer(ByteBuffer buffer) {
        if (buffer.isDirect()) {
            return new MemoryBuffer(ByteBufferUtil.getAddress(buffer) + (long)buffer.position(), buffer.remaining(), buffer);
        }
        int offset = buffer.arrayOffset() + buffer.position();
        return new MemoryBuffer(buffer.array(), offset, buffer.remaining());
    }

    public static MemoryBuffer fromDirectByteBuffer(ByteBuffer buffer, int size, ForyStreamReader streamReader) {
        long offHeapAddress = ByteBufferUtil.getAddress(buffer) + (long)buffer.position();
        return new MemoryBuffer(offHeapAddress, size, buffer, streamReader);
    }

    public static MemoryBuffer fromNativeAddress(long address, int size) {
        return new MemoryBuffer(address, size, null);
    }

    public static MemoryBuffer newHeapBuffer(int initialSize) {
        return MemoryBuffer.fromByteArray(new byte[initialSize]);
    }

    private class BoundChecker
    extends AbstractStreamReader {
        private BoundChecker() {
        }

        @Override
        public int fillBuffer(int minFillSize) {
            throw new IndexOutOfBoundsException(String.format("readerIndex(%d) + length(%d) exceeds size(%d): %s", MemoryBuffer.this.readerIndex, minFillSize, MemoryBuffer.this.size, this));
        }

        @Override
        public MemoryBuffer getBuffer() {
            return MemoryBuffer.this;
        }
    }
}

