/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode;

import java.util.ArrayList;
import java.util.List;
import java.util.TreeSet;
import org.apache.commons.logging.Log;
import org.apache.hadoop.hdfs.protocol.DatanodeInfo;
import org.apache.hadoop.hdfs.protocol.LocatedBlock;
import org.apache.hadoop.hdfs.server.namenode.DatanodeDescriptor;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.hadoop.net.NetworkTopology;
import org.apache.hadoop.net.Node;
import org.apache.hadoop.net.NodeBase;

class ReplicationTargetChooser {
    private final boolean considerLoad;
    private NetworkTopology clusterMap;
    private FSNamesystem fs;

    ReplicationTargetChooser(boolean considerLoad, FSNamesystem fs, NetworkTopology clusterMap) {
        this.considerLoad = considerLoad;
        this.fs = fs;
        this.clusterMap = clusterMap;
    }

    DatanodeDescriptor[] chooseTarget(int numOfReplicas, DatanodeDescriptor writer, List<Node> excludedNodes, long blocksize) {
        if (excludedNodes == null) {
            excludedNodes = new ArrayList<Node>();
        }
        return this.chooseTarget(numOfReplicas, writer, new ArrayList<DatanodeDescriptor>(), excludedNodes, blocksize);
    }

    DatanodeDescriptor[] chooseTarget(int numOfReplicas, DatanodeDescriptor writer, List<DatanodeDescriptor> choosenNodes, List<Node> excludedNodes, long blocksize) {
        if (numOfReplicas == 0 || this.clusterMap.getNumOfLeaves() == 0) {
            return new DatanodeDescriptor[0];
        }
        if (excludedNodes == null) {
            excludedNodes = new ArrayList<Node>();
        }
        int clusterSize = this.clusterMap.getNumOfLeaves();
        int totalNumOfReplicas = choosenNodes.size() + numOfReplicas;
        if (totalNumOfReplicas > clusterSize) {
            numOfReplicas -= totalNumOfReplicas - clusterSize;
            totalNumOfReplicas = clusterSize;
        }
        int maxNodesPerRack = (totalNumOfReplicas - 1) / this.clusterMap.getNumOfRacks() + 2;
        ArrayList<DatanodeDescriptor> results = new ArrayList<DatanodeDescriptor>(choosenNodes);
        excludedNodes.addAll(choosenNodes);
        if (!this.clusterMap.contains(writer)) {
            writer = null;
        }
        DatanodeDescriptor localNode = this.chooseTarget(numOfReplicas, writer, excludedNodes, blocksize, maxNodesPerRack, results);
        results.removeAll(choosenNodes);
        return this.getPipeline(writer == null ? localNode : writer, results.toArray(new DatanodeDescriptor[results.size()]));
    }

    private DatanodeDescriptor chooseTarget(int numOfReplicas, DatanodeDescriptor writer, List<Node> excludedNodes, long blocksize, int maxNodesPerRack, List<DatanodeDescriptor> results) {
        boolean newBlock;
        if (numOfReplicas == 0 || this.clusterMap.getNumOfLeaves() == 0) {
            return writer;
        }
        int numOfResults = results.size();
        boolean bl = newBlock = numOfResults == 0;
        if (writer == null && !newBlock) {
            writer = results.get(0);
        }
        try {
            switch (numOfResults) {
                case 0: {
                    writer = this.chooseLocalNode(writer, excludedNodes, blocksize, maxNodesPerRack, results);
                    if (--numOfReplicas == 0) break;
                }
                case 1: {
                    this.chooseRemoteRack(1, results.get(0), excludedNodes, blocksize, maxNodesPerRack, results);
                    if (--numOfReplicas == 0) break;
                }
                case 2: {
                    if (this.clusterMap.isOnSameRack(results.get(0), results.get(1))) {
                        this.chooseRemoteRack(1, results.get(0), excludedNodes, blocksize, maxNodesPerRack, results);
                    } else if (newBlock) {
                        this.chooseLocalRack(results.get(1), excludedNodes, blocksize, maxNodesPerRack, results);
                    } else {
                        this.chooseLocalRack(writer, excludedNodes, blocksize, maxNodesPerRack, results);
                    }
                    if (--numOfReplicas == 0) break;
                }
                default: {
                    this.chooseRandom(numOfReplicas, "", excludedNodes, blocksize, maxNodesPerRack, results);
                }
            }
        }
        catch (NotEnoughReplicasException e) {
            FSNamesystem.LOG.warn((Object)("Not able to place enough replicas, still in need of " + numOfReplicas));
        }
        return writer;
    }

    private DatanodeDescriptor chooseLocalNode(DatanodeDescriptor localMachine, List<Node> excludedNodes, long blocksize, int maxNodesPerRack, List<DatanodeDescriptor> results) throws NotEnoughReplicasException {
        if (localMachine == null) {
            return this.chooseRandom("", excludedNodes, blocksize, maxNodesPerRack, results);
        }
        if (!excludedNodes.contains(localMachine)) {
            excludedNodes.add(localMachine);
            if (this.isGoodTarget(localMachine, blocksize, maxNodesPerRack, false, results)) {
                results.add(localMachine);
                return localMachine;
            }
        }
        return this.chooseLocalRack(localMachine, excludedNodes, blocksize, maxNodesPerRack, results);
    }

    private DatanodeDescriptor chooseLocalRack(DatanodeDescriptor localMachine, List<Node> excludedNodes, long blocksize, int maxNodesPerRack, List<DatanodeDescriptor> results) throws NotEnoughReplicasException {
        if (localMachine == null) {
            return this.chooseRandom("", excludedNodes, blocksize, maxNodesPerRack, results);
        }
        try {
            return this.chooseRandom(localMachine.getNetworkLocation(), excludedNodes, blocksize, maxNodesPerRack, results);
        }
        catch (NotEnoughReplicasException e1) {
            DatanodeInfo newLocal = null;
            for (DatanodeDescriptor nextNode : results) {
                if (nextNode == localMachine) continue;
                newLocal = nextNode;
                break;
            }
            if (newLocal != null) {
                try {
                    return this.chooseRandom(newLocal.getNetworkLocation(), excludedNodes, blocksize, maxNodesPerRack, results);
                }
                catch (NotEnoughReplicasException e2) {
                    return this.chooseRandom("", excludedNodes, blocksize, maxNodesPerRack, results);
                }
            }
            return this.chooseRandom("", excludedNodes, blocksize, maxNodesPerRack, results);
        }
    }

    private void chooseRemoteRack(int numOfReplicas, DatanodeDescriptor localMachine, List<Node> excludedNodes, long blocksize, int maxReplicasPerRack, List<DatanodeDescriptor> results) throws NotEnoughReplicasException {
        int oldNumOfReplicas = results.size();
        try {
            this.chooseRandom(numOfReplicas, "~" + localMachine.getNetworkLocation(), excludedNodes, blocksize, maxReplicasPerRack, results);
        }
        catch (NotEnoughReplicasException e) {
            this.chooseRandom(numOfReplicas - (results.size() - oldNumOfReplicas), localMachine.getNetworkLocation(), excludedNodes, blocksize, maxReplicasPerRack, results);
        }
    }

    private DatanodeDescriptor chooseRandom(String nodes, List<Node> excludedNodes, long blocksize, int maxNodesPerRack, List<DatanodeDescriptor> results) throws NotEnoughReplicasException {
        DatanodeDescriptor[] selectedNodes;
        DatanodeDescriptor result;
        do {
            if ((selectedNodes = this.chooseRandom(1, nodes, excludedNodes)).length != 0) continue;
            throw new NotEnoughReplicasException("Not able to place enough replicas");
        } while (!this.isGoodTarget(result = selectedNodes[0], blocksize, maxNodesPerRack, results));
        results.add(result);
        return result;
    }

    private void chooseRandom(int numOfReplicas, String nodes, List<Node> excludedNodes, long blocksize, int maxNodesPerRack, List<DatanodeDescriptor> results) throws NotEnoughReplicasException {
        boolean toContinue = true;
        do {
            DatanodeDescriptor[] selectedNodes;
            if ((selectedNodes = this.chooseRandom(numOfReplicas, nodes, excludedNodes)).length < numOfReplicas) {
                toContinue = false;
            }
            for (int i = 0; i < selectedNodes.length; ++i) {
                DatanodeDescriptor result = selectedNodes[i];
                if (!this.isGoodTarget(result, blocksize, maxNodesPerRack, results)) continue;
                --numOfReplicas;
                results.add(result);
            }
        } while (numOfReplicas > 0 && toContinue);
        if (numOfReplicas > 0) {
            throw new NotEnoughReplicasException("Not able to place enough replicas");
        }
    }

    private DatanodeDescriptor[] chooseRandom(int numOfReplicas, String nodes, List<Node> excludedNodes) {
        ArrayList<DatanodeDescriptor> results = new ArrayList<DatanodeDescriptor>();
        int numOfAvailableNodes = this.clusterMap.countNumOfAvailableNodes(nodes, excludedNodes);
        int n = numOfReplicas = numOfAvailableNodes < numOfReplicas ? numOfAvailableNodes : numOfReplicas;
        while (numOfReplicas > 0) {
            DatanodeDescriptor choosenNode = (DatanodeDescriptor)this.clusterMap.chooseRandom(nodes);
            if (excludedNodes.contains(choosenNode)) continue;
            results.add(choosenNode);
            excludedNodes.add(choosenNode);
            --numOfReplicas;
        }
        return results.toArray(new DatanodeDescriptor[results.size()]);
    }

    private boolean isGoodTarget(DatanodeDescriptor node, long blockSize, int maxTargetPerLoc, List<DatanodeDescriptor> results) {
        return this.isGoodTarget(node, blockSize, maxTargetPerLoc, this.considerLoad, results);
    }

    private boolean isGoodTarget(DatanodeDescriptor node, long blockSize, int maxTargetPerLoc, boolean considerLoad, List<DatanodeDescriptor> results) {
        Log logr = FSNamesystem.LOG;
        if (node.isDecommissionInProgress() || node.isDecommissioned()) {
            logr.debug((Object)("Node " + NodeBase.getPath(node) + " is not chosen because the node is (being) decommissioned"));
            return false;
        }
        long remaining = node.getRemaining() - (long)node.getBlocksScheduled() * blockSize;
        if (blockSize * 5L > remaining) {
            logr.debug((Object)("Node " + NodeBase.getPath(node) + " is not chosen because the node does not have enough space"));
            return false;
        }
        if (considerLoad) {
            double avgLoad = 0.0;
            int size = this.clusterMap.getNumOfLeaves();
            if (size != 0) {
                avgLoad = (double)this.fs.getTotalLoad() / (double)size;
            }
            if ((double)node.getXceiverCount() > 2.0 * avgLoad) {
                logr.debug((Object)("Node " + NodeBase.getPath(node) + " is not chosen because the node is too busy"));
                return false;
            }
        }
        String rackname = node.getNetworkLocation();
        int counter = 1;
        for (Node node2 : results) {
            if (!rackname.equals(node2.getNetworkLocation())) continue;
            ++counter;
        }
        if (counter > maxTargetPerLoc) {
            logr.debug((Object)("Node " + NodeBase.getPath(node) + " is not chosen because the rack has too many chosen nodes"));
            return false;
        }
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private DatanodeDescriptor[] getPipeline(DatanodeDescriptor writer, DatanodeDescriptor[] nodes) {
        if (nodes.length == 0) {
            return nodes;
        }
        NetworkTopology networkTopology = this.clusterMap;
        synchronized (networkTopology) {
            int index = 0;
            if (writer == null || !this.clusterMap.contains(writer)) {
                writer = nodes[0];
            }
            while (index < nodes.length) {
                DatanodeDescriptor shortestNode = nodes[index];
                int shortestDistance = this.clusterMap.getDistance(writer, shortestNode);
                int shortestIndex = index;
                for (int i = index + 1; i < nodes.length; ++i) {
                    DatanodeDescriptor currentNode = nodes[i];
                    int currentDistance = this.clusterMap.getDistance(writer, currentNode);
                    if (shortestDistance <= currentDistance) continue;
                    shortestDistance = currentDistance;
                    shortestNode = currentNode;
                    shortestIndex = i;
                }
                if (index != shortestIndex) {
                    nodes[shortestIndex] = nodes[index];
                    nodes[index] = shortestNode;
                }
                writer = shortestNode;
                ++index;
            }
        }
        return nodes;
    }

    public static int verifyBlockPlacement(LocatedBlock lBlk, short replication, NetworkTopology cluster) {
        int numRacks = ReplicationTargetChooser.verifyBlockPlacement(lBlk, Math.min(2, replication), cluster);
        return numRacks < 0 ? 0 : numRacks;
    }

    public static int verifyBlockPlacement(LocatedBlock lBlk, int minRacks, NetworkTopology cluster) {
        int numRacks;
        DatanodeInfo[] locs = lBlk.getLocations();
        if (locs == null) {
            locs = new DatanodeInfo[]{};
        }
        if ((numRacks = cluster.getNumOfRacks()) <= 1) {
            return 0;
        }
        minRacks = Math.min(minRacks, numRacks);
        TreeSet<String> racks = new TreeSet<String>();
        for (DatanodeInfo dn : locs) {
            racks.add(dn.getNetworkLocation());
        }
        return minRacks - racks.size();
    }

    private static class NotEnoughReplicasException
    extends Exception {
        NotEnoughReplicasException(String msg) {
            super(msg);
        }
    }
}

