/*
 * Copyright 2018-2019 Baidu, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software distributed under the License is distributed on
 * an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 */
package com.baidubce.services.dugo;

import static com.baidubce.util.Validate.checkStringNotEmpty;

import java.net.URI;
import java.util.HashMap;
import java.util.Map;

import com.baidubce.AbstractBceClient;
import com.baidubce.BceClientConfiguration;
import com.baidubce.http.HttpMethodName;
import com.baidubce.http.handler.BceErrorResponseHandler;
import com.baidubce.http.handler.BceJsonResponseHandler;
import com.baidubce.http.handler.BceMetadataResponseHandler;
import com.baidubce.http.handler.HttpResponseHandler;
import com.baidubce.internal.InternalRequest;
import com.baidubce.services.dugo.alarm.AlarmBatchRequest;
import com.baidubce.services.dugo.alarm.AlarmCreateRequest;
import com.baidubce.services.dugo.alarm.AlarmDetailsRequest;
import com.baidubce.services.dugo.alarm.AlarmRuleListRequest;
import com.baidubce.services.dugo.alarm.AlarmUpdateRequest;
import com.baidubce.services.dugo.batch.BindRequest;
import com.baidubce.services.dugo.batch.QueryInstancesByBatchRequest;
import com.baidubce.services.dugo.batch.VehicleBindBatchRequest;
import com.baidubce.services.dugo.cert.DownloadAuthInfoRequest;
import com.baidubce.services.dugo.cert.QueryBindResultRequest;
import com.baidubce.services.dugo.core.protocol.http.BceIotHttpClient;
import com.baidubce.services.dugo.dynamicdata.DynamicPropertyRequest;
import com.baidubce.services.dugo.map.DrivingBehaviorRequest;
import com.baidubce.services.dugo.map.GeoCodingRequest;
import com.baidubce.services.dugo.map.GetDistanceRequest;
import com.baidubce.services.dugo.map.GetLatestPointRequest;
import com.baidubce.services.dugo.map.GetTrackRequest;
import com.baidubce.services.dugo.map.ReverseGeoCodingRequest;
import com.baidubce.services.dugo.map.StayPointRequest;
import com.baidubce.services.dugo.project.GetBatchsRequest;
import com.baidubce.services.dugo.project.GetProjectByIdRequest;
import com.baidubce.services.dugo.project.GetProjectListRequest;
import com.baidubce.services.dugo.project.UpdateBatchDesRequest;
import com.baidubce.services.dugo.shadow.MultipleShadowRequest;
import com.baidubce.services.dugo.shadow.QueryVehicleStatusRequest;
import com.baidubce.services.dugo.shadow.ShadowFilterRequest;
import com.baidubce.services.dugo.shadow.SingleShadowRequest;
import com.baidubce.services.dugo.staticdata.StaticDataRequest;
import com.baidubce.util.HttpUtils;
import com.baidubce.util.JsonUtils;

/**
 * The client for dugo.
 * Created by liuzhenxing01 on 2018/11/5.
 */
public class DuGoClient extends AbstractBceClient {
    private static final String ENDPOINT = "https://ivc.gz.baidubce.com";
    private static final String PROJECT_ID = "projectId";
    private static final String PREFIX_VERSION1 = "/v1/ivc/data";

    private static HttpResponseHandler[] duGoHandlers = new HttpResponseHandler[] {
            new BceMetadataResponseHandler(),
            new BceErrorResponseHandler(),
            new BceJsonResponseHandler()
    };

    public DuGoClient(BceClientConfiguration config) {
        super(!config.getEndpoint().equals(ENDPOINT) ? config.withEndpoint(ENDPOINT) : config, duGoHandlers);
    }

    /**
     * @param request Dynamic properties
     * upload Dynamic Property
     * @return http status-code is : 204 no content. if return null, request failed.
     * uploadDynamicProperty (eg . color)
     * @OpenAPI-uri eg: https://ivc.gz.baidubce.com/v1/data
     */
    public DynamicPropertyRequest.DynamicPropertyResponse uploadDynamicProperty(DynamicPropertyRequest request) {
        InternalRequest internalRequest = buildInternalRequest("/v1/data", HttpMethodName.POST, request);
        return this.invokeHttpClient(internalRequest, DynamicPropertyRequest.DynamicPropertyResponse.class);
    }

    /**
     * @param request
     * upload Static Data
     * @return http status-code is : 200 empty content return.
     * @OpenAPI-uri eg: https://ivc.gz.baidubce.com/v1/ivc/data/static/upload
     */
    public StaticDataRequest.StaticDataResponse uploadStaticData(StaticDataRequest request) {
        isEmptyCheck(request.getVehicleId(), "vehicleId");
        InternalRequest internalRequest =
                buildInternalRequest(PREFIX_VERSION1 + "/static/upload", HttpMethodName.POST, request);
        return this.invokeHttpClient(internalRequest, StaticDataRequest.StaticDataResponse.class);
    }

    /**
     * @param request model for single shoadow
     * query Single Shadow
     * @return SingleShadowResponse
     * @OpenAPI-uri eg: https://ivc.gz.baidubce.com/v1/ivc/data/shadow/query
     */
    public SingleShadowRequest.SingleShadowResponse querySingleShadow(SingleShadowRequest request) {
        isEmptyCheck(request.getVehicleId(), "vehicleId");
        InternalRequest internalRequest =
                buildInternalRequest(PREFIX_VERSION1 + "/shadow/query", HttpMethodName.POST, request);
        return this.invokeHttpClient(internalRequest, SingleShadowRequest.SingleShadowResponse.class);
    }

    /**
     * @param request
     * query Multiple Shadow
     * @return
     * @OpenAPI-uri eg: https://ivc.gz.baidubce.com/v1/ivc/data/shadow/batchquery
     */
    public MultipleShadowRequest.MultipleShadowResponse queryMultipleShadow(MultipleShadowRequest request) {
        if (request == null || request.getVehicleIds().size() == 0) {
            throw new IllegalArgumentException("vehicleIds should not be empty");
        }
        InternalRequest internalRequest =
                buildInternalRequest(PREFIX_VERSION1 + "/shadow/batchquery", HttpMethodName.POST, request);
        return this.invokeHttpClient(internalRequest, MultipleShadowRequest.MultipleShadowResponse.class);
    }

    /**
     * @param request
     * query Shadow By Tag
     * @return
     * @OpenAPI-uri eg: https://ivc.gz.baidubce.com/v1/ivc/data/shadow/filterquery
     */
    public ShadowFilterRequest.ShadowFilterResponse queryShadowByTag(ShadowFilterRequest request) {
        isEmptyCheck(request.getProjectId(), PROJECT_ID);
        InternalRequest internalRequest =
                buildInternalRequest(PREFIX_VERSION1 + "/shadow/filterquery", HttpMethodName.POST, request);
        return this.invokeHttpClient(internalRequest, ShadowFilterRequest.ShadowFilterResponse.class);
    }

    /**
     * address -> （latitude， longitude）
     * geo Coding
     * @param request
     * @return
     * @OpenAPI-uri eg: https://ivc.gz.baidubce.com/v1/ivc/data/geocoder
     */
    public GeoCodingRequest.GeoCodingResponse geoCoding(GeoCodingRequest request) {
        isEmptyCheck(request.getAddress(), "address");
        InternalRequest internalRequest = buildInternalRequest(PREFIX_VERSION1 + "/geocoder",
                HttpMethodName.POST, request);
        return this.invokeHttpClient(internalRequest, GeoCodingRequest.GeoCodingResponse.class);
    }

    /**
     * （latitude， longitude）-> address
     * @param request
     * reverse Geo Coding
     * @return
     * @OpenAPI-uri eg: https://ivc.gz.baidubce.com/v1/ivc/data/reversegeocoder
     */
    public ReverseGeoCodingRequest.ReverseGeoCodingResponse reverseGeoCoding(ReverseGeoCodingRequest request) {
        InternalRequest internalRequest =
                buildInternalRequest(PREFIX_VERSION1 + "/reversegeocoder", HttpMethodName.POST, request);
        return this.invokeHttpClient(internalRequest, ReverseGeoCodingRequest.ReverseGeoCodingResponse.class);
    }

    /**
     * lastPoint
     * get Latest Point
     * @param request
     * @OpenAPI-uri eg: https://ivc.gz.baidubce.com/v1/ivc/data/track/getlatestpoint
     */
    public GetLatestPointRequest.GetLatestPointResponse getLatestPoint(GetLatestPointRequest request) {
        isEmptyCheck(request.getVehicleId(), "vehicleId");
        InternalRequest internalRequest =
                buildInternalRequest(PREFIX_VERSION1 + "/track/getlatestpoint", HttpMethodName.POST, request);
        return this.invokeHttpClient(internalRequest, GetLatestPointRequest.GetLatestPointResponse.class);
    }

    /**
     * @param request
     * get Distance
     * @return
     * @OpenAPI-uri eg: https://ivc.gz.baidubce.com/v1/ivc/data/track/getdistance
     */
    public GetDistanceRequest.GetDistanceResponse getDistance(GetDistanceRequest request) {
        isEmptyCheck(request.getVehicleId(), "vehicleId");
        InternalRequest internalRequest =
                buildInternalRequest(PREFIX_VERSION1 + "/track/getdistance", HttpMethodName.POST, request);
        return this.invokeHttpClient(internalRequest, GetDistanceRequest.GetDistanceResponse.class);
    }

    /**
     * @param request
     * get Track
     * @return
     * @OpenAPI-uri eg: https://ivc.gz.baidubce.com/v1/ivc/data/track/gettrack
     */
    public GetTrackRequest.GetTrackResponse getTrack(GetTrackRequest request) {
        isEmptyCheck(request.getVehicleId(), "vehicleId");
        InternalRequest internalRequest =
                buildInternalRequest(PREFIX_VERSION1 + "/track/gettrack", HttpMethodName.POST, request);
        return this.invokeHttpClient(internalRequest, GetTrackRequest.GetTrackResponse.class);
    }

    /**
     * @param request
     * get Stay Point
     * @return
     * @OpenAPI-uri eg: https://ivc.gz.baidubce.com/v1/ivc/data/analysis/staypoint
     */
    public StayPointRequest.StayPointResponse getStayPoint(StayPointRequest request) {
        isEmptyCheck(request.getVehicleId(), "vehicleId");
        InternalRequest internalRequest =
                buildInternalRequest(PREFIX_VERSION1 + "/analysis/staypoint", HttpMethodName.POST, request);
        return this.invokeHttpClient(internalRequest, StayPointRequest.StayPointResponse.class);
    }

    /**
     * @param request
     * get driving behavior
     * @return
     * @OpenAPI-uri eg: https://ivc.gz.baidubce.com/v1/ivc/data/analysis/drivingbehavior
     */
    public DrivingBehaviorRequest.DrivingBehaviorResponse getDrivingBehavior(DrivingBehaviorRequest request) {
        isEmptyCheck(request.getVehicleId(), "vehicleId");
        InternalRequest internalRequest =
                buildInternalRequest(PREFIX_VERSION1 + "/analysis/drivingbehavior", HttpMethodName.POST, request);
        return this.invokeHttpClient(internalRequest, DrivingBehaviorRequest.DrivingBehaviorResponse.class);
    }

    /**
     * @param request
     * random Bind Vehicle To Project
     * @return
     * @OpenAPI-uri eg:  https://ivc.gz.baidubce.com/v1/ivc/data/vehicle/bind
     */
    public BindRequest.BindResponse randomBindVehicleToProject(BindRequest request) {
        isEmptyCheck(request.getProjectId(), PROJECT_ID);
        InternalRequest internalRequest =
                buildInternalRequest(PREFIX_VERSION1 + "/vehicle/bind", HttpMethodName.POST, request);
        return this.invokeHttpClient(internalRequest, BindRequest.BindResponse.class);
    }

    /**
     * @param request
     * unbind Vehicle
     * @return
     * @OpenAPI-uri eg: https://ivc.gz.baidubce.com/v1/ivc/data/vehicle/unbind
     */
    public BindRequest.BindResponse unbindVehicle(BindRequest request) {
        isEmptyCheck(request.getProjectId(), PROJECT_ID);
        InternalRequest internalRequest =
                buildInternalRequest(PREFIX_VERSION1 + "/vehicle/unbind", HttpMethodName.POST, request);
        return this.invokeHttpClient(internalRequest, BindRequest.BindResponse.class);
    }

    /**
     * @return
     * get Alarm Details
     * @OpenAPI-uri eg: https://ivc.gz.baidubce.com/v1/ivc/data/alarm?projectId=?&alarmId=?
     */
    public AlarmDetailsRequest.AlarmDetailsResponse getAlarmDetails(String projectId, String alarmId) {
        isEmptyCheck(projectId, PROJECT_ID);
        isEmptyCheck(alarmId, "alarmId");
        Map<String, String> param = new HashMap<String, String>();
        param.put(PROJECT_ID, projectId);
        param.put("alarmId", alarmId);
        InternalRequest internalRequest = buildInternalRequest(PREFIX_VERSION1 + "/alarm",
                HttpMethodName.GET, null, param);
        return this.invokeHttpClient(internalRequest, AlarmDetailsRequest.AlarmDetailsResponse.class);
    }

    /**
     * get Alarms
     * @param projectId
     * @param pageNum
     * @param pageSize
     * @return
     */
    public AlarmRuleListRequest.AlarmRuleListResponse getAlarms(String projectId, int pageNum, int pageSize) {
        isEmptyCheck(projectId, PROJECT_ID);
        Map<String, String> param = new HashMap<String, String>();
        param.put(PROJECT_ID, projectId);
        param.put("pageNum", String.valueOf(pageNum));
        param.put("pageSize", String.valueOf(pageSize));
        param.put("list", "");
        InternalRequest internalRequest = buildInternalRequest(PREFIX_VERSION1 + "/alarm",
                HttpMethodName.GET, null, param);
        return this.invokeHttpClient(internalRequest, AlarmRuleListRequest.AlarmRuleListResponse.class);
    }

    /**
     * @return
     * @param projectId
     * @OpenAPI-uri eg: https://ivc.gz.baidubce.com/v1/ivc/data/alarm?projectId=..
     */
    public AlarmBatchRequest.AlarmBatchResponse batchDelete(AlarmBatchRequest request, String projectId) {
        isEmptyCheck(projectId, PROJECT_ID);
        Map<String, String> param = new HashMap<String, String>();
        param.put("delete", "");
        param.put(PROJECT_ID, projectId);
        InternalRequest internalRequest =
                buildInternalRequest(PREFIX_VERSION1 + "/alarm", HttpMethodName.POST, request, param);
        return this.invokeHttpClient(internalRequest, AlarmBatchRequest.AlarmBatchResponse.class);
    }

    /**
     * @param projectId
     * @return
     * batch Shield
     * @OpenAPI-uri eg: https://ivc.gz.baidubce.com/v1/ivc/data/alarm?projectId=...
     */
    public AlarmBatchRequest.AlarmBatchResponse batchShield(AlarmBatchRequest request, String projectId) {
        isEmptyCheck(projectId, PROJECT_ID);
        Map<String, String> param = new HashMap<String, String>();
        param.put("shield", "");
        param.put(PROJECT_ID, projectId);
        InternalRequest internalRequest =
                buildInternalRequest(PREFIX_VERSION1 + "/alarm", HttpMethodName.POST, request, param);
        return this.invokeHttpClient(internalRequest, AlarmBatchRequest.AlarmBatchResponse.class);
    }

    /**
     * @param projectId
     * batch Recover
     * @OpenAPI-uri eg: https://ivc.gz.baidubce.com/v1/ivc/data/alarm?projectId=...
     */
    public AlarmBatchRequest.AlarmBatchResponse batchRecover(AlarmBatchRequest request, String projectId) {
        isEmptyCheck(projectId, PROJECT_ID);
        Map<String, String> param = new HashMap<String, String>();
        param.put("recover", "");
        param.put(PROJECT_ID, projectId);
        InternalRequest internalRequest =
                buildInternalRequest(PREFIX_VERSION1 + "/alarm", HttpMethodName.POST, request, param);
        return this.invokeHttpClient(internalRequest, AlarmBatchRequest.AlarmBatchResponse.class);
    }

    /**
     * @param projectId
     * @param request
     * create AlarmRule
     * @return
     * @OpenAPI-uri eg: https://ivc.gz.baidubce.com/v1/ivc/data/alarm?projectId=...
     * method POST
     */
    public AlarmCreateRequest.AlarmCreateResponse createAlarmRule(AlarmCreateRequest request, String projectId) {
        isEmptyCheck(projectId, PROJECT_ID);
        Map<String, String> param = new HashMap<String, String>();
        param.put(PROJECT_ID, projectId);
        InternalRequest internalRequest =
                buildInternalRequest(PREFIX_VERSION1 + "/alarm", HttpMethodName.POST, request, param);
        return this.invokeHttpClient(internalRequest, AlarmCreateRequest.AlarmCreateResponse.class);
    }

    /**
     * @param projectId
     * @param request
     * update Alarm
     * @OpenAPI-uri eg: https://ivc.gz.baidubce.com/v1/ivc/data/alarm?projectId=...
     * method = PUT
     */
    public AlarmUpdateRequest.AlarmUpdateResponse updateAlarm(AlarmUpdateRequest request, String projectId) {
        isEmptyCheck(projectId, PROJECT_ID);
        Map<String, String> param = new HashMap<String, String>();
        param.put(PROJECT_ID, projectId);
        InternalRequest internalRequest =
                buildInternalRequest(PREFIX_VERSION1 + "/alarm", HttpMethodName.PUT, request, param);
        return this.invokeHttpClient(internalRequest, AlarmUpdateRequest.AlarmUpdateResponse.class);
    }

    /**
     *
     * @return
     * get Projects
     * @OpenAPI-uri eg: https://ivc.gz.baidubce.com/v1/ivc/data/project
     */
    public GetProjectListRequest.GetProjectListResponse getProjects() {
        InternalRequest internalRequest = buildInternalRequest(PREFIX_VERSION1 + "/project", HttpMethodName.GET, null,
                null);
        return this.invokeHttpClient(internalRequest, GetProjectListRequest.GetProjectListResponse.class);
    }

    /**
     * @param projectId
     * @return
     * get Batchs
     * @OpenAPI-uri eg: https://ivc.gz.baidubce.com/v1/ivc/data/{projectId}/batches
     */
    public GetBatchsRequest.GetBatchsResponse getBatchs(String projectId) {
        isEmptyCheck(projectId, PROJECT_ID);
        InternalRequest internalRequest =
                buildInternalRequest(PREFIX_VERSION1 + "/project/" + projectId + "/batches", HttpMethodName.GET, null,
                        null);
        return this.invokeHttpClient(internalRequest, GetBatchsRequest.GetBatchsResponse.class);
    }

    /**
     * @return
     * get Project By Id
     * @OpenAPI-uri eg: https://ivc.gz.baidubce.com/v1/ivc/data/{projectId}
     */
    public GetProjectByIdRequest.GetProjectByIdResponse getProjectById(String projectId) {
        isEmptyCheck(projectId, PROJECT_ID);
        InternalRequest internalRequest =
                buildInternalRequest(PREFIX_VERSION1 + "/project/" + projectId, HttpMethodName.GET, null, null);
        return this.invokeHttpClient(internalRequest, GetProjectByIdRequest.GetProjectByIdResponse.class);
    }

    /**
     * @return
     * @param projectId
     * @param batchId
     * update Batch Des
     * @OpenAPI-uri eg: https://ivc.gz.baidubce.com/v1/ivc/data/{projectId}/batch/{batchId}
     */
    public UpdateBatchDesRequest.UpdateBatchDesResponse updateBatchDes(UpdateBatchDesRequest request, String
            projectId, String batchId) {
        isEmptyCheck(projectId, PROJECT_ID);
        isEmptyCheck(batchId, "batchId");
        InternalRequest internalRequest =
                buildInternalRequest(PREFIX_VERSION1 + "/project/" + projectId + "/batch/" + batchId,
                        HttpMethodName.PUT, request, null);
        return this.invokeHttpClient(internalRequest, UpdateBatchDesRequest.UpdateBatchDesResponse.class);
    }

    /**
     * @return
     * query Bind Result
     * @OpenAPI-uri eg: https://ivc.gz.baidubce.com/v1/ivc/data/mqtt/results?projectId=...
     */
    public QueryBindResultRequest.QueryBindResultResponse queryBindResult(String projectId) {
        isEmptyCheck(projectId, PROJECT_ID);
        Map<String, String> param = new HashMap<String, String>();
        param.put(PROJECT_ID, projectId);
        InternalRequest internalRequest =
                buildInternalRequest(PREFIX_VERSION1 + "/mqtt/results", HttpMethodName.GET, null, param);
        return this.invokeHttpClient(internalRequest, QueryBindResultRequest.QueryBindResultResponse.class);
    }

    /**
     * @return
     * @downloadUrl
     * @OpenAPI-uri eg: https://ivc.gz.baidubce.com/v1/ivc/data/mqtt/download?downloadUrl=...
     */
    public DownloadAuthInfoRequest.DownloadAuthInfoResponse downloadAuthInfo(String downloadUrl) {
        isEmptyCheck(downloadUrl, "downloadUrl");
        Map<String, String> param = new HashMap<String, String>();
        param.put("downloadUrl", downloadUrl);
        InternalRequest internalRequest =
                buildInternalRequest(PREFIX_VERSION1 + "/mqtt/download", HttpMethodName.GET, null, param);
        return this.invokeHttpClient(internalRequest, DownloadAuthInfoRequest.DownloadAuthInfoResponse.class);
    }

    /**
     * @param request
     *
     * @return
     *
     * @OpenAPI-uri eg: https://ivc.gz.baidubce.com/v1/ivc/data/vehicle/bind/batch
     */
    public VehicleBindBatchRequest.VehicleBindBatchResponse bindVehiclesInBatch(VehicleBindBatchRequest request) {
        isEmptyCheck(request.getBatchId(), "batchId");
        InternalRequest internalRequest =
                buildInternalRequest(PREFIX_VERSION1 + "/vehicle/bind/batch", HttpMethodName.POST, request, null);
        return this.invokeHttpClient(internalRequest, VehicleBindBatchRequest.VehicleBindBatchResponse.class);
    }

    /**
     * @return
     * @pageSize
     * @pageNum
     * query Instance By Batch
     * @OpenAPI-uri eg: https://ivc.gz.baidubce.com/v1/ivc/data/vehicle/batch
     */
    public QueryInstancesByBatchRequest.QueryInstancesByBatchResponse queryInstanceByBatch(String batchId, int
            pageNum, int pageSize) {
        isEmptyCheck(batchId, "batchId");
        Map<String, String> param = new HashMap<String, String>();
        param.put("batchId", batchId);
        param.put("pageNum", String.valueOf(pageNum));
        param.put("pageSize", String.valueOf(pageSize));
        InternalRequest internalRequest =
                buildInternalRequest(PREFIX_VERSION1 + "/vehicle/batch", HttpMethodName.GET, null, param);
        return this.invokeHttpClient(internalRequest, QueryInstancesByBatchRequest.QueryInstancesByBatchResponse.class);
    }

    /**
     * @param request
     * query Vehicle Online Status
     * @return
     * @OpenAPI-uri eg: https://ivc.gz.baidubce.com/v1/ivc/data/shadow/onlinequery
     */
    public QueryVehicleStatusRequest.QueryVehicleStatusResponse queryVehicleOnlineStatus(QueryVehicleStatusRequest
                                                                                                 request) {
        InternalRequest internalRequest =
                buildInternalRequest(PREFIX_VERSION1 + "/shadow/onlinequery", HttpMethodName.POST, request);
        return this.invokeHttpClient(internalRequest, QueryVehicleStatusRequest.QueryVehicleStatusResponse.class);
    }

    private InternalRequest buildInternalRequest(String path, HttpMethodName methodName, AbstractRequest request,
                                                 Map<String, String> param) {
        BceIotHttpClient client = new BceIotHttpClient();
        client.withAuth(config.getCredentials().getAccessKeyId(), config.getCredentials().getSecretKey());
        URI uri = HttpUtils.appendUri(this.getEndpoint(), path);
        client.withMethod(methodName, uri);
        if (methodName == HttpMethodName.POST || methodName == HttpMethodName.PUT) {
            client.withPayload(JsonUtils.toJsonString(request).getBytes());
        }

        if (param != null) {
            for (Map.Entry<String, String> entry : param.entrySet()) {
                client.addParams(entry.getKey(), entry.getValue());
            }
        }
        return client.getInternalRequest();
    }

    private InternalRequest buildInternalRequest(String path, HttpMethodName methodName, AbstractRequest request) {
        return buildInternalRequest(path, methodName, request, null);
    }

    private void isEmptyCheck(String keyValue, String keyName) {
        checkStringNotEmpty(keyValue, keyName + " should not be null");
    }
}
