package com.chinamcloud.common.cache.client;

import com.chinamcloud.common.cache.constant.CacheConstant;
import lombok.extern.slf4j.Slf4j;
import net.rubyeye.xmemcached.MemcachedClient;
import net.rubyeye.xmemcached.exception.MemcachedException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.concurrent.TimeoutException;

/**
 * Created with IntelliJ IDEA.
 * User: chenzhiwei
 * Date: 17/4/27
 * Time: 下午1:36
 */
@Slf4j
@Component("memcachedCommonClient")
public class MemcachedCommonClient {

    @Autowired
    private MemcachedClient memcachedClient;

    /**
     * 保存到缓存
     * @param key：缓存的键
     * @param expireTime：过期时间,单位秒
     * @param value：缓存的值
     * @return
     */
    public boolean set(String key, Object value, Integer expireTime){
        try {
            return memcachedClient.set(key,expireTime,value);
        } catch (InterruptedException e) {
            log.error("缓存数据遇到异常,key:"+key,e);
        } catch (MemcachedException e) {
            log.error("缓存数据遇到异常,key:"+key,e);
        } catch (TimeoutException e) {
            log.error("缓存数据遇到异常,key:"+key,e);
        }
        return false;
    }

    /**
     * 保存到缓存,默认30天
     * @param key：缓存的键
     * @param value：缓存的值
     * @return
     */
    public boolean set(String key, Object value){
        try {
            return memcachedClient.set(key,CacheConstant.ONE_WEEK_EXPIRED_TIME,value);
        } catch (InterruptedException e) {
            log.error("缓存数据遇到异常,key:"+key,e);
        } catch (MemcachedException e) {
            log.error("缓存数据遇到异常,key:"+key,e);
        } catch (TimeoutException e) {
            log.error("缓存数据遇到异常,key:"+key,e);
        }
        return false;
    }

    /**
     * 获得缓存key相关的值
     * @param key
     * @return
     */
    public Object get(String key) {
        try {
            return memcachedClient.get(key);
        } catch (TimeoutException e) {
            log.error("获取缓存数据遇到异常,key:"+key,e);
        } catch (InterruptedException e) {
            log.error("获取缓存数据遇到异常,key:"+key,e);
        } catch (MemcachedException e) {
            log.error("获取缓存数据遇到异常,key:"+key,e);
        }
        return null;
    }

    /**
     * 删除缓存数据
     * @param key
     * @return
     */
    public boolean delete(String key){
        try {
            return memcachedClient.delete(key);
        } catch (TimeoutException e) {
            log.error("删除缓存数据遇到异常,key:"+key,e);
        } catch (InterruptedException e) {
            log.error("删除缓存数据遇到异常,key:"+key,e);
        } catch (MemcachedException e) {
            log.error("删除缓存数据遇到异常,key:"+key,e);
        }
        return false;
    }

    /**
     * 指定key是否存在
     * @param key
     * @return
     */
    public boolean isExist(String key) {
        return null != get(key);
    }

    /**
     * 使所有连接的memcached的数据项无效
     */
    public boolean flushAll() {
        try {
            memcachedClient.flushAll();
            return true;
        } catch (TimeoutException e) {
            log.error("所有缓存数据置为无效遇到异常:",e);
        } catch (InterruptedException e) {
            log.error("所有缓存数据置为无效遇到异常:",e);
        } catch (MemcachedException e) {
            log.error("所有缓存数据置为无效遇到异常:",e);
        }
        return false;
    }

    /**
     * 指定的key自增
     * @param key
     * @param delta:增量
     * @param initValue:初始值
     * @return
     */
    public Long incr(String key,Long delta,Long initValue){
        return incr(key,delta,initValue, CacheConstant.DEFAULT_TIMEOUT, CacheConstant.ONE_WEEK_EXPIRED_TIME);
    }


    /**
     * 指定的key自增
     * @param key
     * @param delta:增量
     * @param initValue:初始值
     * @return
     */
    public Long incr(String key,Long delta,Long initValue,Long timeout,Integer expireTime){
        try {
            return memcachedClient.incr(key,delta,initValue,timeout,expireTime);
        } catch (TimeoutException e) {
            log.error("缓存数据自增遇到异常:",e);
        } catch (InterruptedException e) {
            log.error("缓存数据自增遇到异常:",e);
        } catch (MemcachedException e) {
            log.error("缓存数据自增遇到异常:",e);
        }
        throw new RuntimeException("缓存数据自增遇到异常") ;
    }

    /**
     * 指定的key自减
     * @param key
     * @param delta:增量
     * @param initValue:初始值
     * @return
     */
    public Long decr(String key,Long delta,Long initValue){
        return decr(key,delta,initValue,CacheConstant.DEFAULT_TIMEOUT, CacheConstant.ONE_WEEK_EXPIRED_TIME);
    }

    /**
     * 指定的key自减
     * @param key
     * @param delta:增量
     * @param initValue:初始值
     * @return
     */
    public Long decr(String key,Long delta,Long initValue,Long timeout,Integer expireTime){
        try {
            return memcachedClient.incr(key,delta,initValue,timeout,expireTime);
        } catch (TimeoutException e) {
            log.error("缓存数据自增遇到异常:",e);
        } catch (InterruptedException e) {
            log.error("缓存数据自增遇到异常:",e);
        } catch (MemcachedException e) {
            log.error("缓存数据自增遇到异常:",e);
        }
        throw new RuntimeException("缓存数据自增遇到异常") ;
    }

}
