package com.chinamclound.common.cache.client;

import com.chinamcloud.common.cache.constant.CacheConstant;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.BoundSetOperations;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.ValueOperations;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.Set;
import java.util.concurrent.TimeUnit;

/**
 * Created with IntelliJ IDEA.
 * User: chenzhiwei
 * Date: 17/5/18
 * Time: 下午5:05
 */
@Component("redisByLongCommonClient")
public class RedisByLongCommonClient<K, V> {
    @Autowired
    @Resource(name = "longRedisTemplate")
    private RedisTemplate<K, V> longRedisTemplate;
    @Resource(name = "longRedisTemplate")
    private ValueOperations<K, V> longValueOperations;

    /**
     * 保存到缓存
     *
     * @param key：缓存的键
     * @param value：缓存的值
     * @param expireTime：过期时间,单位秒
     * @return
     * @see CacheConstant
     */
    public boolean set(K key, V value, Integer expireTime) {
        longValueOperations.set(key, value, expireTime.longValue(), TimeUnit.SECONDS);
        return true;
    }

    /**
     * 保存到缓存,默认1天
     *
     * @param key：缓存的键
     * @param value：缓存的值
     * @return
     */
    public boolean set(K key, V value) {
        return set(key, value, CacheConstant.ONE_DAY_EXPIRED_TIME);
    }

    /**
     * 获得缓存key相关的值
     *
     * @param key
     * @return
     */
    public V get(K key) {
        return longValueOperations.get(key);
    }

    /**
     * 指定的key自增
     *
     * @param key
     * @param delta:增量
     * @param expireTime:过期时间，单位秒
     * @return
     * @see com.chinamcloud.common.cache.constant.CacheConstant
     */
    public Long increment(K key, Long delta, Integer expireTime) {
        Long incrementSum = longValueOperations.increment(key, delta);
        longRedisTemplate.expire(key, expireTime, TimeUnit.SECONDS);
        return incrementSum;
    }

    /**
     * 重置为0，并返回原来的值
     *
     * @param key
     * @return
     */
    public V deleteAndReturnValue(K key) {
        V cacheValue = longValueOperations.get(key);
        longRedisTemplate.delete(key);
        return cacheValue;
    }

    /**
     * 数据缓存在set集合中
     *
     * @param key
     * @param value
     * @param expireTime:过期时间，单位秒
     * @see com.chinamcloud.common.cache.constant.CacheConstant
     */
    public void setCacheSet(K key, V value, Integer expireTime) {
        BoundSetOperations<K, V> setOperations = longRedisTemplate.boundSetOps(key);
        setOperations.add(value);
        setOperations.expire(expireTime.longValue(), TimeUnit.SECONDS);
    }

    /**
     * 返回set集合中的数据
     *
     * @param key
     */
    public Set<V> getCacheSet(K key) {
        BoundSetOperations<K, V> setOperations = longRedisTemplate.boundSetOps(key);
        return setOperations.members();
    }

    /**
     * 抛出一个值
     *
     * @param key
     */
    public V cacheSetPop(K key) {
        BoundSetOperations<K, V> setOperations = longRedisTemplate.boundSetOps(key);
        return setOperations.pop();
    }

    /**
     * 返回set尺寸的大小
     *
     * @param key
     */
    public Long cacheSetSize(K key) {
        BoundSetOperations<K, V> setOperations = longRedisTemplate.boundSetOps(key);
        return setOperations.size();
    }

}
