package com.chinamclound.common.cache.client;

import com.google.common.collect.Maps;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.BoundHashOperations;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * Created with IntelliJ IDEA.
 * User: chenzhiwei
 * Date: 17/5/11
 * Time: 下午1:44
 * 参考：http://www.cnblogs.com/0201zcr/p/4987561.html
 */
@Component("redisMapByLongCommonClient")
public class RedisMapByLongCommonClient<K, HK, HV> {
    @Autowired
    @Resource(name = "longRedisTemplate")
    private RedisTemplate<K, Object> longRedisTemplate;

    /**
     * 保存Map数据到指定的key中
     *
     * @param cacheKey
     * @param cacheMap
     * @param expireTime:过期时间，单位秒
     * @return
     * @see com.chinamcloud.common.cache.constant.CacheConstant
     */
    public boolean putCacheMap(K cacheKey, Map<HK, HV> cacheMap, Integer expireTime) {
        BoundHashOperations<K, HK, HV> hashOperations = longRedisTemplate.boundHashOps(cacheKey);
        hashOperations.putAll(cacheMap);
        hashOperations.expire(expireTime, TimeUnit.SECONDS);
        return true;
    }

    /**
     * 返回指定key的Map中的相应key数据（只返回存在的数据）
     *
     * @param cacheKey
     */
    public Map<HK, HV> getCacheMap(K cacheKey, List<HK> mapKeyList) {
        BoundHashOperations<K, HK, HV> hashOperations = longRedisTemplate.boundHashOps(cacheKey);
        List<HV> mapValueList = hashOperations.multiGet(mapKeyList);
        Map<HK, HV> resultMap = Maps.newHashMap();
        for (int i = 0; i < mapValueList.size(); i++) {
            HV mapValue = mapValueList.get(i);
            if (mapValue != null) {
                resultMap.put(mapKeyList.get(i), mapValue);
            }
        }
        return resultMap;
    }


    /**
     * 删除map中指定的key
     *
     * @param cacheKey
     * @param mapKeyList
     * @return :删除成功的个数
     */
    public Long delete(K cacheKey, List<HK> mapKeyList) {
        BoundHashOperations<K, HK, HV> hashOperations = longRedisTemplate.boundHashOps(cacheKey);
        return hashOperations.delete(mapKeyList.toArray());
    }

    /**
     * map中指定的key增加指定的增量
     *
     * @param cacheKey
     * @param mapKey
     * @param delta:增量
     * @param expireTime:过期时间，单位秒
     * @return
     * @see com.chinamcloud.common.cache.constant.CacheConstant
     */
    public Long increment(K cacheKey, HK mapKey, Long delta, Integer expireTime) {
        BoundHashOperations<K, HK, HV> hashOperations = longRedisTemplate.boundHashOps(cacheKey);
        hashOperations.expire(expireTime, TimeUnit.SECONDS);
        return hashOperations.increment(mapKey, delta);
    }

    /**
     * 删除并返回原存储的值
     *
     * @param cacheKey
     * @param mapKey
     * @return 返回删除的key的值
     */
    public HV deleteAndReturnValue(final K cacheKey, final HK mapKey) {
        BoundHashOperations<K, HK, HV> hashOperations = longRedisTemplate.boundHashOps(cacheKey);
        HV mapValue = hashOperations.get(mapKey);
        hashOperations.delete(mapKey);
        return mapValue;
    }
}
