package com.chinamclound.common.cache.client;

import com.chinamcloud.common.cache.constant.CacheConstant;
import com.chinamclound.common.cache.properties.RedisCustomizeProperties;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.*;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.Set;
import java.util.concurrent.TimeUnit;

/**
 * Created with IntelliJ IDEA.
 * User: chenzhiwei
 * Date: 17/5/11
 * Time: 下午1:44
 * 参考：http://www.cnblogs.com/0201zcr/p/4987561.html
 */
@Component("redisZsetCommonClient")
public class RedisZSetCommonClient<K, V> {

    //Spring Boot已为我们配置RedisTemplate，在此处可以直接注入
    @Autowired
    @Resource(name = "redisTemplate")
    private RedisTemplate<K, V> redisTemplate;
    //可以使用@Resource注解指定redisTmplate，可以注入基于对象的简单属性操作
    @Resource(name = "redisTemplate")
    private ValueOperations<K, V> valueOperations;
    @Autowired
    private RedisCustomizeProperties redisCustomizeProperties;


    /**
     * 删除缓存数据
     *
     * @param key
     * @return
     */
    public void delete(K key) {
        redisTemplate.delete(key);
    }

    /**
     * 指定key是否存在
     *
     * @param key
     * @return
     */
    public boolean isExist(K key) {
        return redisTemplate.hasKey(key);
    }


    /**
     * 数据缓存在zset集合中
     *
     * @param key
     * @param min
     * @param max
     * @see CacheConstant
     */
    public Long countZSet(K key, double min,double max) {
        BoundZSetOperations<K, V> kvBoundZSetOperations = redisTemplate.boundZSetOps(key);
        return kvBoundZSetOperations.count(min,max);
    }

    /**
     * 数据缓存在set集合中
     *
     * @param key
     * @param value
     * @see CacheConstant
     */
    public Long removeCacheZSetByValues(K key, V ... value) {
        BoundZSetOperations<K, V> kvBoundZSetOperations = redisTemplate.boundZSetOps(key);
        return kvBoundZSetOperations.remove(value);
    }


    /**
     * 数据缓存在set集合中
     *
     * @param key
     * @param min
     * @param max
     * @see CacheConstant
     */
    public void removeCacheZSetByScore(K key, double min, double max) {
        BoundZSetOperations<K, V> kvBoundZSetOperations = redisTemplate.boundZSetOps(key);
        kvBoundZSetOperations.removeRangeByScore(min,max);
    }

    /**
     * 数据缓存在zset集合中
     *
     * @param key
     * @param value
     * @see CacheConstant
     */
    public void setCacheZSet(K key, V value,double score) {
        setCacheZSet(key,value,score,redisCustomizeProperties.getDefaultExpiration());
    }


    /**
     * 数据缓存在zset集合中
     *
     * @param key
     * @param value
     * @param expireTime:过期时间，单位秒
     * @see CacheConstant
     */
    public void setCacheZSet(K key, V value,double score, Integer expireTime) {
        BoundZSetOperations<K, V> kvBoundZSetOperations = redisTemplate.boundZSetOps(key);
        kvBoundZSetOperations.add(value,score);
        kvBoundZSetOperations.expire(expireTime.longValue(), TimeUnit.SECONDS);
    }




    /**
     * 数据缓存在zset集合中
     *
     * @param key
     * @param tuples
     * @param expireTime:过期时间，单位秒
     * @see CacheConstant
     */
    public void setCacheZSet(K key, Set<ZSetOperations.TypedTuple<V>> tuples, Integer expireTime) {
        BoundZSetOperations<K, V> kvBoundZSetOperations = redisTemplate.boundZSetOps(key);
        kvBoundZSetOperations.add(tuples);
        kvBoundZSetOperations.expire(expireTime.longValue(), TimeUnit.SECONDS);
    }

    /**
     * 数据缓存在zset集合中
     *
     * @param key
     * @param tuples
     * @see CacheConstant
     */
    public void setCacheZSet(K key, Set<ZSetOperations.TypedTuple<V>> tuples) {
        setCacheZSet(key,tuples,redisCustomizeProperties.getDefaultExpiration());
    }


    /**
     * 增加指定V的分数
     *
     * @param key
     * @param value
     * @param score
     * @return 返回增加后的值
     */
    public Double incrementScoreZSet(K key, V value,double score) {
        BoundZSetOperations<K, V> kvBoundZSetOperations = redisTemplate.boundZSetOps(key);
        return kvBoundZSetOperations.incrementScore(value,score);
    }

    /**
     * 获取指定值的分数
     * @param key
     * @param value
     * @return :返回实际分数值
     */
    public Double scoreZSet(K key, V value) {
        BoundZSetOperations<K, V> kvBoundZSetOperations = redisTemplate.boundZSetOps(key);
        return kvBoundZSetOperations.score(value);
    }

    /**
     * 返回zset集合中的数据
     *
     * @param key
     */
    public Set<V> getCacheZSet(K key,double min,double max) {
        BoundZSetOperations<K, V> kvBoundZSetOperations = redisTemplate.boundZSetOps(key);
        return kvBoundZSetOperations.rangeByScore(min,max);
    }
    /**
     * 返回zset集合中的数据
     *
     * @param key
     */
    public Set<V> getCacheZSetByReverse(K key,double min,double max) {
        BoundZSetOperations<K, V> kvBoundZSetOperations = redisTemplate.boundZSetOps(key);
        return kvBoundZSetOperations.reverseRangeByScore(min,max);
    }


    /**
     * 返回zset集合中的数据
     *
     * @param key
     */
    public Set<ZSetOperations.TypedTuple<V>> getCacheWithScoresZSet(K key, double min, double max) {
        BoundZSetOperations<K, V> kvBoundZSetOperations = redisTemplate.boundZSetOps(key);
        return kvBoundZSetOperations.rangeByScoreWithScores(min,max);
    }



}
