package com.chinamclound.common.datasource.multiple;

import com.alibaba.druid.pool.DruidDataSource;
import com.alibaba.fastjson.JSONObject;
import com.chinamcloud.common.result.ResultDTO;
import com.chinamcloud.common.util.OkHttpUtil;
import com.chinamclound.common.datasource.constant.DataSourceConstant;
import com.chinamclound.common.datasource.properties.AuthProperties;
import com.chinamclound.common.datasource.properties.DatabaseProperties;
import com.chinamclound.common.datasource.transformer.DatabaseProperties2DruidDataSource;
import com.google.common.base.MoreObjects;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.datasource.lookup.AbstractRoutingDataSource;
import org.springframework.util.StringUtils;

import javax.sql.DataSource;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Created with IntelliJ IDEA.
 * User: chenzhiwei
 * Date: 16/12/23
 * Time: 下午6:50
 * 创建动态数据源类：继承org.springframework.jdbc.datasource.lookup.AbstractRoutingDataSource
 *
 */
@Slf4j
public class DynamicDataSource extends AbstractRoutingDataSource {
    //存放数据库连接信息（存放格式，key-value）
    private static  Map<Object, Object> recordTargetDataSources =new ConcurrentHashMap<Object,Object>();
    //默认数据库连接信息
    private static DataSource recordDefaultTargetDataSource;

    @Autowired
    private AuthProperties authProperties;
    @Autowired
    private DatabaseProperties databaseProperties;

    public void setRecordTargetDataSources(Map<Object, Object> duplicateTargetDataSources) {
        this.recordTargetDataSources = duplicateTargetDataSources;
    }


    public void setRecordDefaultTargetDataSource(DataSource duplicateDefaultTargetDataSource) {
        this.recordDefaultTargetDataSource = duplicateDefaultTargetDataSource;
    }

    /**
     * 切换到租户数据源
     * @param tenantId
     */
    public ResultDTO<Boolean> switchDataSourceByTenantId(String tenantId) {
        if(recordTargetDataSources.keySet().contains(tenantId) || DataSourceConstant.DEFAULT_TENANTID.equals(tenantId) ){
            DataSourceHolder.setDataSource(tenantId);
            return ResultDTO.successfy(true);
        }
        DatabaseProperties databaseProperties = queryDataBasePropertiesByTenantId(tenantId);
        if(databaseProperties == null){
            return ResultDTO.fail("查询不到租户数据库信息");
        }
        DruidDataSource dataSource = DatabaseProperties2DruidDataSource.INSTANCE.apply(databaseProperties);
        this.recordTargetDataSources.put(tenantId,dataSource);
        super.setTargetDataSources(this.recordTargetDataSources);
        //设置AbstractRoutingDataSource参数后要调用afterPropertiesSet()方法，spring容器才会进行加载操作
        super.afterPropertiesSet();
        DataSourceHolder.setDataSource(tenantId);
        return ResultDTO.successfy(true);
    }
    /**
     * 切换到默认数据源
     */
    public ResultDTO<Boolean> switchDefaultDataSource() {
        return switchDataSourceByTenantId(DataSourceConstant.DEFAULT_TENANTID);
    }


    private DatabaseProperties queryDataBasePropertiesByTenantId(String tenantId) {
        Map<String,String> parameterMap = Maps.newHashMap();
        parameterMap.put("tenantid", tenantId);
        parameterMap.put("appname", authProperties.getCenter_appname());
        parameterMap.put("secret", authProperties.getCenter_secret());
        String response = OkHttpUtil.get(authProperties.getCenter_url(), null, parameterMap);
        if(StringUtils.isEmpty(response)){
            return null;
        }
        try {
            JSONObject jsonObject = JSONObject.parseObject(response);
            Integer returnCode = jsonObject.getInteger("returnCode");
            if (0 != returnCode) {
                log.error("请求权限中心，返回码失败:{}", returnCode);
                return null;
            }
            JSONObject dbJsonObject = jsonObject.getJSONObject("returnData").getJSONObject("db_info");
            ResultDTO<Boolean> booleanResultDTO = checkParameter(dbJsonObject);
            if (!booleanResultDTO.isSuccess()) {
                log.error(booleanResultDTO.getDescription());
                return null;
            }
            DatabaseProperties databasePropertiesByQuery = getDatabaseProperties(dbJsonObject);
            return databasePropertiesByQuery;
        }catch (Exception e){
            log.warn(e.getMessage());
            return null;
        }
    }

    private DatabaseProperties getDatabaseProperties(JSONObject dbJsonObject) {
        DatabaseProperties databasePropertiesByQuery = new DatabaseProperties();
        StringBuilder stringBuilder = new StringBuilder();
        stringBuilder.append("jdbc:mysql://").append(dbJsonObject.getString("host")).append(":").append(
                MoreObjects.firstNonNull(dbJsonObject.getString("port"),"3306")).append("/").
                append(dbJsonObject.getString("database")).append("?useUnicode=true&characterEncoding=utf8");
        databasePropertiesByQuery.setUrl(stringBuilder.toString());
        databasePropertiesByQuery.setUsername(dbJsonObject.getString("login"));
        databasePropertiesByQuery.setPassword(dbJsonObject.getString("password"));
        databasePropertiesByQuery.setFilters(databaseProperties.getFilters());
        databasePropertiesByQuery.setConnectionProperties(databaseProperties.getConnectionProperties());
        return databasePropertiesByQuery;
    }

    private ResultDTO<Boolean> checkParameter(JSONObject jsonObject) {
        if(StringUtils.isEmpty(jsonObject.getString("host"))){
            return ResultDTO.fail("host is empty");
        }
        if(StringUtils.isEmpty(jsonObject.getString("database"))){
            return ResultDTO.fail("database is empty");
        }
        if(StringUtils.isEmpty(jsonObject.getString("login"))){
            return ResultDTO.fail("login is empty");
        }
        if(StringUtils.isEmpty(jsonObject.getString("password"))){
            return ResultDTO.fail("password is empty");
        }
        return ResultDTO.successfy(true);
    }

    /**
     * 返回当前数据源标识符：只需要重写方法
     * @return
     */
    @Override
    protected Object determineCurrentLookupKey() {
        return DataSourceHolder.getDataSource();
    }
}
