package com.chinamcloud.spider.base;


import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

/**
 * 分页查询返回结果集
 * @param <T> 实体对象
 * Created by jyy on 17/6/6.
 */
public class PageResult<T> implements Serializable {

    //模式分页
    public static int DEFAULT_PAGE_SIZE = 10;

    //当前页
    private int currentPage;

    //每页条数
    private int pageSize ;

    //总记录数
    private int totalRecords;

    //总页数
    private int pageCount;

    //开始页数
    private int startRecord;

    //是否有上一页
    private boolean hasPreviousPage;

    //是否有下一页
    private boolean hasNextPage;

    //是否只有一页
    private boolean onlyOnePage;

    //结果集
    private List<T> pageRecords;

    //排序
    private String orderField = "create_time";
    private String orderDirection = "DESC";

    public int getCurrentPage() {
        return currentPage;
    }

    public PageResult() {
        this(0, 0,null,null);
    }

    public PageResult(int currentPage, int pageSize) {
        this(currentPage, pageSize,null,null);
    }

    /**
     * 构造方法，只构造空页.
     */
    public PageResult(int currentPage, int pageSize, String orderField, String orderDirection) {
        this(currentPage, 0, pageSize, new ArrayList(), orderField, orderDirection);
    }

    /**
     * 默认构造方法.
     *
     * @param currentPage     本页数据在数据库中的起始位置
     * @param totalSize 数据库中总记录条数
     * @param pageSize  本页容量
     * @param pageRecords      本页包含的数据
     */
    public PageResult(int currentPage, int totalSize, int pageSize, List<T> pageRecords, String orderField, String orderDirection) {
        this.pageSize = pageSize;
        this.currentPage = currentPage;
        this.totalRecords = totalSize;
        this.pageRecords = pageRecords;
        if (orderField != null)
            this.orderField = orderField;
        if (orderDirection != null)
            this.orderDirection = orderDirection;
        if (this.pageSize == 0) {
            this.pageSize = DEFAULT_PAGE_SIZE;
        }
    }


    /**
     * make sure the page is in the range of the total pages
     *
     * @param currentPage current page
     */
    public void setCurrentPage(int currentPage) {
        if (currentPage < 1) {
            this.currentPage = 1;
            return;
        }
        if (currentPage > getPageCount()) {
            this.currentPage = getPageCount();
            return;
        }
        this.currentPage = currentPage;
    }

    public String getOrderField() {
        return orderField;
    }

    public void setOrderField(String orderField) {
        this.orderField = orderField;
    }

    public String getOrderDirection() {
        return orderDirection;
    }

    public void setOrderDirection(String orderDirection) {
        this.orderDirection = orderDirection;
    }

    /**
     * get page size
     *
     * @return page size number
     */
    public int getPageSize() {
        return pageSize;
    }

    /**
     * set page size
     *
     * @param pageSize page size number
     */
    public void setPageSize(int pageSize) {
        if (pageSize <= 0) {
            this.pageSize = 1;
        } else {
            this.pageSize = pageSize;
        }
    }

    /**
     * get total records
     *
     * @return total record's number
     */
    public int getTotalRecords() {
        return totalRecords;
    }

    /**
     * set total records
     *
     * @param totalRecords total record number
     */
    public void setTotalRecords(int totalRecords) {
        this.totalRecords = totalRecords;
    }

    /**
     * Get the total count of the page
     *
     * @return count number
     */
    public int getPageCount() {
        // If there is no data in database.
        if (totalRecords == 0) {
            pageCount = 1;
            return pageCount;
        }
        boolean isZero = totalRecords % pageSize == 0;
        pageCount = totalRecords / pageSize;
        pageCount = isZero ? pageCount : pageCount + 1;
        return pageCount;
    }

    /**
     * First record of one page
     *
     * @return start records
     */
    public int getStartRecord() {
        if (currentPage < 1)
            currentPage = 1;
        startRecord = ((currentPage - 1) * pageSize);
        return startRecord;
    }

    /**
     * Whether has previous page
     *
     * @return if previous page's is exist,return true else not
     */
    public boolean isHasPreviousPage() {
        hasPreviousPage = (currentPage == 1) ? false : true;
        return hasPreviousPage;
    }

    /**
     * Whether has next page
     *
     * @return if next page's is exist,return true else not
     */
    public boolean isHasNextPage() {
        hasNextPage = (currentPage == getPageCount()) ? false : true;
        return hasNextPage;
    }

    /**
     * Whether is only one page
     *
     * @return if only one page,return true else not
     */
    public boolean isOnlyOnePage() {
        onlyOnePage = ((getPageCount() == 1) ? true : false);
        return onlyOnePage;
    }

    /**
     * @return the pageRecords
     */
    public List<T> getPageRecords() {
        return pageRecords;
    }

    /**
     * @param pageRecords the pageRecords to set
     */
    public void setPageRecords(List<T> pageRecords) {
        this.pageRecords = pageRecords;
    }

    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("{pageSize=")
                .append(pageSize)
                .append(", currentPage=")
                .append(currentPage)
                .append(", totalSize=")
                .append(totalRecords);
        if (totalRecords > 0) {
            sb.append(",pageRecords size =")
                    .append(pageRecords.size());
        }
        sb.append("}");
        return sb.toString();
    }
}