package com.chinamcloud.spider.utils;

import com.alibaba.fastjson.JSONObject;
import com.chinamcloud.spider.dto.InterfaceDto;
import com.chinamcloud.spider.dto.MultipartBodyPartDto;
import lombok.extern.slf4j.Slf4j;
import okhttp3.*;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.io.IOUtils;

import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * Created with IntelliJ IDEA.
 * User: chenzhiwei
 * Date: 18/2/24
 * Time: 下午2:56
 */
@Slf4j
public class OkHttpUtil2 {
    /**
     * get 方式提交
     *
     * @param url          : 请求的url
     * @param headerMap    : 请求头
     * @param parameterMap : 请求参数
     * @return
     */
    public static Response get(String url, Map<String, String> headerMap, Map<String, Object> parameterMap) {
        OkHttpClient okHttpClient = new OkHttpClient();
        Request request = getRequest(url, headerMap, parameterMap);
        try {
            return okHttpClient.newCall(request).execute();
        } catch (IOException e) {
            log.error("请求遇到异常", e);
            throw new RuntimeException(e);
        }
    }

    private static Request getRequest(String url, Map<String, String> headerMap, Map<String, Object> parameterMap) {
        String completeUrl = getCompleteUrl(url, parameterMap);
        Request.Builder builder = new Request.Builder();
        if (MapUtils.isNotEmpty(headerMap)) {
            for (String key : headerMap.keySet()) {
                builder.addHeader(key, headerMap.get(key));
            }
        }
        return builder.url(completeUrl).build();
    }

    private static String getCompleteUrl(String url, Map<String, Object> parameterMap) {
        if (MapUtils.isEmpty(parameterMap)) {
            return url;
        }
        StringBuilder stringBuilder = new StringBuilder(url).append("?");
        int index = 0;
        for (String key : parameterMap.keySet()) {
            Object o = parameterMap.get(key);
            if (o == null) {
                continue;
            }
            if (index > 0) {
                stringBuilder.append("&");
            }
            stringBuilder.append(key).append("=").append(o.toString());
            index++;
        }
        return stringBuilder.toString();
    }

    /**
     * 以json格式post提交
     *
     * @param url       : 请求的url
     * @param headerMap : 请求头
     * @param parameter : Json格式的数据
     * @return
     */
    public static Response postByJson(String url, Map<String, String> headerMap, String parameter) {
        MediaType MEDIA_TYPE_TEXT = MediaType.parse("application/json");
        RequestBody requestBody = RequestBody.create(MEDIA_TYPE_TEXT, parameter);
        return post(url, headerMap, requestBody);
    }

    private static Response post(String url, Map<String, String> headerMap, RequestBody requestBody) {
        OkHttpClient okHttpClient = new OkHttpClient();
        Request.Builder builder = new Request.Builder();
        if (MapUtils.isNotEmpty(headerMap)) {
            for (String key : headerMap.keySet()) {
                builder.addHeader(key, headerMap.get(key));
            }
        }
        Request request = builder.url(url).post(requestBody).build();
        try {
            return okHttpClient.newCall(request).execute();
        } catch (Exception e) {
            log.error("请求遇到异常", e);
            throw new RuntimeException(e);
        }
    }

    /**
     * 以form表单的形式提交
     *
     * @param url          : 请求的url
     * @param headerMap    : 请求头
     * @param parameterMap : 请求参数，key:value
     * @return
     */
    public static Response postByForm(String url, Map<String, String> headerMap, Map<String, Object> parameterMap) {
        FormBody.Builder formBodyBuilder = new FormBody.Builder();
        if (MapUtils.isNotEmpty(parameterMap)) {
            for (String key : parameterMap.keySet()) {
                Object o = parameterMap.get(key);
                if (o == null) {
                    continue;
                }
                formBodyBuilder.add(key, o.toString());
            }
        }
        RequestBody requestBody = formBodyBuilder.build();
        return post(url, headerMap, requestBody);

    }


    public static Response uploadFile(String url, Map<String, String> headerMap, Map<String, Object> parameterMap, List<MultipartBodyPartDto> multipartBodyPartDtoList) {
        try {
            MultipartBody.Builder multipartBodyBuilder = new MultipartBody.Builder().setType(MultipartBody.FORM);
            if (MapUtils.isNotEmpty(parameterMap)) {
                for (Map.Entry<String, Object> entry : parameterMap.entrySet()) {
                    Object o = entry.getValue();
                    if (o == null) {
                        continue;
                    }
                    multipartBodyBuilder.addFormDataPart(entry.getKey(), o.toString());
                }
            }
            Request.Builder builder = new Request.Builder();
            if (MapUtils.isNotEmpty(headerMap)) {
                for (Map.Entry<String, String> entry : headerMap.entrySet()) {
                    builder.addHeader(entry.getKey(), entry.getValue());
                }
            }
            if (CollectionUtils.isNotEmpty(multipartBodyPartDtoList)) {
                for (MultipartBodyPartDto multipartBodyPartDto : multipartBodyPartDtoList) {
                    multipartBodyBuilder.addPart(MultipartBody.Part.createFormData(multipartBodyPartDto.getName(), multipartBodyPartDto.getOriginalFilename(),
                            RequestBody.create(MediaType.parse("text/plain"), IOUtils.toByteArray(multipartBodyPartDto.getInputStream()))));
                }
            }
            RequestBody requestBody = multipartBodyBuilder.build();
            OkHttpClient okHttpClient = new OkHttpClient.Builder().
                    connectTimeout(500, TimeUnit.SECONDS).
                    writeTimeout(500, TimeUnit.SECONDS).
                    readTimeout(500, TimeUnit.SECONDS).
                    build();
            Request request = builder.url(url).post(requestBody).build();
            return okHttpClient.newCall(request).execute();
        } catch (IOException e) {
            log.error("请求遇到异常", e);
            throw new RuntimeException(e);
        }
    }

    public static Response execute(InterfaceDto interfaceDto) {
        Map<String, String> headerMap = interfaceDto.getHeaderMap();
        String method = headerMap.get("method");
        Response response = null;
        if ("get".equalsIgnoreCase(method)) {
            response = OkHttpUtil2.get(interfaceDto.getUrl(), interfaceDto.getHeaderMap(), interfaceDto.getParameterMap());
        } else {
            String contentType = headerMap.get("contentType");
            if (contentType != null && contentType.toLowerCase().contains("application/json")) {
                JSONObject jsonObject = new JSONObject();
                jsonObject.putAll(interfaceDto.getParameterMap());
                response = OkHttpUtil2.postByJson(interfaceDto.getUrl(), interfaceDto.getHeaderMap(), jsonObject.toJSONString());
            } else if (CollectionUtils.isNotEmpty(interfaceDto.getMultipartBodyPartDtoList())) {
                response = OkHttpUtil2.uploadFile(interfaceDto.getUrl(), interfaceDto.getHeaderMap(), interfaceDto.getParameterMap(), interfaceDto.getMultipartBodyPartDtoList());
            } else {
                response = OkHttpUtil2.postByForm(interfaceDto.getUrl(), interfaceDto.getHeaderMap(), interfaceDto.getParameterMap());
            }
        }
        return response;
    }

    public static String request(InterfaceDto interfaceDto) {
        Response response = execute(interfaceDto);
        if (response == null) {
            return null;
        }
        try {
            String result = response.body().string();
            if (!response.isSuccessful()) {
                log.error("请求interfaceDto:{},返回:{}", interfaceDto, result);
            } else {
                log.debug("请求interfaceDto:{},返回:{}", interfaceDto, result);
            }
            return result;
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }
}
