package com.chinamcloud.spider.system.config;

import com.alibaba.druid.filter.logging.Slf4jLogFilter;
import com.alibaba.druid.pool.DruidDataSource;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.Getter;
import lombok.Setter;
import org.apache.ibatis.session.SqlSessionFactory;
import org.mybatis.spring.SqlSessionFactoryBean;
import org.mybatis.spring.SqlSessionTemplate;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.io.ClassPathResource;
import org.springframework.core.io.Resource;
import org.springframework.core.io.support.PathMatchingResourcePatternResolver;
import org.springframework.jdbc.datasource.DataSourceTransactionManager;
import org.springframework.transaction.annotation.EnableTransactionManagement;

import java.util.Map;
import java.util.Properties;

/**
 * Created by jyy on 17/6/6.
 */
@Configuration
//启用注解事务管理，使用CGLib代理
@EnableTransactionManagement(proxyTargetClass = true)
@Setter
@Getter
@ComponentScan(basePackages = "com.chinamcloud.spider")
public class DataSourceAutoConfiguration {

    //分布式环境各种加载环境配置
    @Value("${spring.druid.datasource.type}")
    private String type;

    @Value("${spring.druid.datasource.driverClassName}")
    private String driverClassName;

    @Value("${spring.druid.datasource.url}")
    private String url;

    @Value("${spring.druid.datasource.username}")
    private String username;

    @Value("${spring.druid.datasource.password}")
    private String password;

    @Value("${spring.druid.datasource.initialSize}")
    private Integer initialSize;

    @Value("${spring.druid.datasource.minIdle}")
    private Integer minIdle;

    @Value("${spring.druid.datasource.maxActive}")
    private Integer maxActive;

    @Value("${spring.druid.datasource.maxWait}")
    private Long maxWait;

    @Value("${spring.druid.datasource.timeBetweenEvictionRunsMillis}")
    private Long timeBetweenEvictionRunsMillis;

    @Value("${spring.druid.datasource.minEvictableIdleTimeMillis}")
    private Long minEvictableIdleTimeMillis;

    @Value("${spring.druid.datasource.validationQuery}")
    private String validationQuery;

    @Value("${spring.druid.datasource.testWhileIdle}")
    private boolean testWhileIdle;

    @Value("${spring.druid.datasource.testOnBorrow}")
    private boolean testOnBorrow;

    @Value("${spring.druid.datasource.testOnReturn}")
    private boolean testOnReturn;

    @Value("${spring.druid.datasource.poolPreparedStatements}")
    private boolean poolPreparedStatements;

    @Value("${spring.druid.datasource.maxPoolPreparedStatementPerConnectionSize}")
    private Integer maxPoolPreparedStatementPerConnectionSize;

    @Value("${spring.druid.datasource.filters}")
    private String filters;

    @Value("${spring.druid.datasource.connectionProperties}")
    private String connectionProperties;

    @Value("${spring.druid.datasource.useGlobalDataSourceStat:false}")
    private boolean useGlobalDataSourceStat;

    private String mapperPath = "**/*Mapper.xml";

    @Bean("spiderDruidDataSource")
    public DruidDataSource dataSource() throws Exception {
        DruidDataSource dataSource = new DruidDataSource();
        dataSource.setDriverClassName(driverClassName);
        dataSource.setUrl(url);
        dataSource.setUsername(username);
        dataSource.setPassword(password);
        dataSource.setInitialSize(initialSize);
        dataSource.setMinIdle(minIdle);
        dataSource.setMaxActive(maxActive);
        dataSource.setMaxWait(maxWait);
        dataSource.setTimeBetweenEvictionRunsMillis(timeBetweenEvictionRunsMillis);
        dataSource.setMinEvictableIdleTimeMillis(minEvictableIdleTimeMillis);
        if (validationQuery != null && !"".equals(validationQuery)) {
            dataSource.setValidationQuery(validationQuery);
        }
        dataSource.setTestWhileIdle(testWhileIdle);
        dataSource.setTestOnBorrow(testOnBorrow);
        dataSource.setTestOnReturn(testOnReturn);
        dataSource.setMaxPoolPreparedStatementPerConnectionSize(maxPoolPreparedStatementPerConnectionSize);
        dataSource.setFilters(filters);//这是最关键的,否则SQL监控无法生效
        if(connectionProperties != null && !"".equals(connectionProperties)){
            Properties connectProperties = new Properties();
            String[] propertiesList = connectionProperties.split(";");
            for(String propertiesTmp:propertiesList){
                String[] obj = propertiesTmp.split("=");
                String key = obj[0];
                String value = obj[1];
                connectProperties.put(key,value);
            }
            dataSource.setConnectProperties(connectProperties);
        }
        dataSource.getProxyFilters().add(slf4jLogFilter());
        dataSource.setUseGlobalDataSourceStat(useGlobalDataSourceStat);
        dataSource.setConnectionInitSqls(Lists.newArrayList("set names utf8mb4;"));
        return dataSource;
    }

    @Bean(name = "spiderSqlSessionFactory")
    public SqlSessionFactory sqlSessionFactoryBean(@Qualifier("spiderDruidDataSource") DruidDataSource druidDataSource) throws Exception {
        SqlSessionFactoryBean sqlSessionFactoryBean = new SqlSessionFactoryBean();
        sqlSessionFactoryBean.setDataSource(druidDataSource);
        PathMatchingResourcePatternResolver resolver = new PathMatchingResourcePatternResolver();
        Resource[] resourceArray = resolver.getResources("classpath*:" + mapperPath);
        Map<String,Resource> resourceMap = Maps.newHashMap();
        for (Resource resource : resourceArray){
            String url = resource.getURL().getPath().replaceAll("\\\\", "/").replaceFirst("file:/","file:");
            if (!resourceMap.containsKey(url)){
                resourceMap.put(url,resource);
            }
        }
        Resource[] resources = new Resource[resourceMap.size()];
        int i= 0;
        for (Map.Entry<String,Resource> entry : resourceMap.entrySet()){
            resources[i] = entry.getValue();
            i++;
        }
        sqlSessionFactoryBean.setMapperLocations(resources);
        sqlSessionFactoryBean.setConfigLocation(new ClassPathResource("mybatis-config.xml"));

        return sqlSessionFactoryBean.getObject();
    }

    @Bean("spiderDataSourceTransactionManager")
    public DataSourceTransactionManager transactionManager(@Qualifier("spiderDruidDataSource") DruidDataSource druidDataSource) throws Exception {
        return new DataSourceTransactionManager(druidDataSource);
    }

    @Bean
    public Slf4jLogFilter  slf4jLogFilter(){
        Slf4jLogFilter slf4jLogFilter = new Slf4jLogFilter();
        slf4jLogFilter.setStatementExecutableSqlLogEnable(true);
        slf4jLogFilter.setConnectionLogEnabled(true);
        slf4jLogFilter.setResultSetLogErrorEnabled(true);
        slf4jLogFilter.setStatementLogEnabled(true);
        return slf4jLogFilter;
    }

    @Bean(name = "spiderSqlSessionTemplate")
    public SqlSessionTemplate SqlSessionTemplate(@Qualifier("spiderSqlSessionFactory") SqlSessionFactory sqlSessionFactory) throws Exception {
        return new SqlSessionTemplate(sqlSessionFactory);
    }

}
