package com.chinamcloud.spider.base;

import com.chinamcloud.spider.exception.DataAccessException;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.ibatis.mapping.ResultMapping;
import org.mybatis.spring.SqlSessionTemplate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Repository;
import org.springframework.util.StringUtils;

import java.io.Serializable;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Created by jyy on 17/6/6.
 */
@Slf4j
@Repository
public  abstract class BaseDao<E, PK extends Serializable>  implements EntityDao<E,PK> {


    @Autowired
    @Qualifier("spiderSqlSessionTemplate")
    protected SqlSessionTemplate sqlSessionTemplate;


    private static DBDriverAdapter dbda = new DBDriverAdapter();

    /**
     * 插入实体对象
     * @param entity the entity
     */
    @Override
    public int save(E entity) {
        log.debug("--------->保存对象 : "+entity);
        return sqlSessionTemplate.insert(getInsertStatement(),entity);
    }

    @Override
    public void batchSave(List<E> entities) throws DataAccessException {
        if (CollectionUtils.isEmpty(entities)){
            log.error("批量保存参数传递错误");
            throw new RuntimeException("批量保存参数传递错误:entities不能为空");
        }
        Map map = new HashMap();
        map.put("list",entities);
        log.debug("--------->批量保存对象 : "+entities);
        sqlSessionTemplate.insert(getBatchInsertStatement(),entities);

    }

    /**
     * 分页查询
     * @param pageRequest the page request 请求参数
     * @return 返回分页结果集
     */
    @Override
    public PageResult findPage(PageRequest pageRequest)  {
        try{
            return pageQuery(sqlSessionTemplate, getPageStatementForPaging(),getCountStatementForPaging(), pageRequest);
        } catch (Exception e) {
            e.printStackTrace();
            throw new DataAccessException("数据操作失败，错误信息："+e.getMessage());
        }

    }
    /**
     * 分页查询
     * @param sqlSessionTemplate
     * @param statementName
     * @param countStatementName
     * @param pageRequest
     * @return
     * @throws Exception
     */
    public static PageResult pageQuery(SqlSessionTemplate sqlSessionTemplate, String statementName,
                                 String countStatementName,
                                 PageRequest pageRequest) throws IllegalAccessException, NoSuchMethodException, InvocationTargetException {
        PageResult pageResult = new PageResult(pageRequest.getPageNumber(), pageRequest.getPageSize());
        Number totalCount = (Number) sqlSessionTemplate.selectOne(
                countStatementName, PropertyUtils.describe(pageRequest));
        pageResult.setTotalRecords(totalCount.intValue());
        pageResult.setOrderField(pageRequest.getOrderField());
        pageResult.setOrderDirection(pageRequest.getOrderDirection());
        log.debug("---------------> : 一共{}数据 ",totalCount);
        if (totalCount != null && totalCount.longValue() > 0) {
            Map filters = new HashMap();
            filters.put("offset", pageResult.getStartRecord());
            filters.put("pageSize", pageResult.getPageSize());
            filters.put("pageS", dbda.getPageS(Constants.DBTYPE));//暂时无用
            filters.put("pageE", dbda.getPageE(Constants.DBTYPE));
            Map parameterObject = PropertyUtils.describe(pageRequest);
            filters.putAll(parameterObject);
            String orderField = getOrderField(sqlSessionTemplate, pageResult.getOrderField(),statementName);
            filters.put("orderField",orderField);
            filters.put("orderDirection", pageResult.getOrderDirection());
            List list = sqlSessionTemplate.selectList(statementName, filters);
            if(!list.isEmpty()){
                pageResult.setPageRecords(list);
            }
        }
        return pageResult;
    }

    private static String getOrderField(SqlSessionTemplate sqlSessionTemplate, String orderField, String statementName) {
        if (StringUtils.isEmpty(orderField)){
            return null;
        }
        String replaceOrderField = null;
        String nameSpace = statementName.substring(0, statementName.lastIndexOf("."));
        List<ResultMapping> resultMappingList = sqlSessionTemplate.getConfiguration().getResultMap(nameSpace + ".BaseResultMap").getResultMappings();
        Map<String, String> columnMap = Maps.newHashMap();
        for (ResultMapping resultMapping : resultMappingList){
            columnMap.put(resultMapping.getProperty(),resultMapping.getColumn());
        }
        String[] fieldArray = orderField.split(",");
        StringBuffer stringBuffer = new StringBuffer();
        for (String field : fieldArray) {
            field = field.trim();
            if (field.contains(" ")) {
                String[] subField = field.split(" ");
                stringBuffer.append(columnMap.get(subField[0].trim())).append(" ").append(subField[1]).append(",");
            } else {
                stringBuffer.append(columnMap.get(field)).append(",");
            }
        }
        if (stringBuffer.length() > 1) {
            replaceOrderField = stringBuffer.toString().substring(0, stringBuffer.length() - 1);
        }
        return replaceOrderField;
    }



    /**
     * 通过主键查询信息
     * @param primaryKey 主键
     * @return
     */
    @Override
    public E getById(PK primaryKey){
        log.debug("----------> 通过主键查询对象 : "+ primaryKey);
        Map<String,Object> paramMap = new HashMap<String, Object>();
        paramMap.put("id", primaryKey);
        E object = sqlSessionTemplate.selectOne(getByIdStatement(), paramMap);
        return object;

    }

    @Override
    public List<E> getByIdList(List<PK> idList){
        Map map = new HashMap();
        map.put("ids", idList);
        return sqlSessionTemplate.selectList(getListByIdsStatement(), map);
    }


    /**
     * 通过主键修改数据
     * @param entity the entity
     */
    @Override
    public void updateById(E entity) {
        log.debug("----------> 通过主键更新数据: "+ entity);
       int count =  sqlSessionTemplate.update(getUpdateStatement(),entity);
        log.debug("--------------> 更新的数据 {} 条",count);
    }

    /**
     *  通过主键删除数据
     * @param id the id
     */
    @Override
    public int deleteById(PK id) {
        log.debug("----------> 通过主键删除数据 : "+ id);
        Map map = new HashMap();
        map.put("id", id);
        int affectCount = sqlSessionTemplate.delete(getDeleteStatement(), map);
        log.debug("--------------> 删除数据 {} 条",affectCount);
        return affectCount;
    }

    /**
     * 通过主键数组删除数据 数据格式 , 分割
     * @param checkIds the check ids
     */
    @Override
    public void deleteByIds(String checkIds) {
        log.debug("----------> 通过主键删除数据 : "+ checkIds);
        if (StringUtils.isEmpty(checkIds)){
            return;
        }
        Map map = new HashMap();
        List<Long> longs = new ArrayList<Long>();
        for (String id: checkIds.split(",")){
            if (!StringUtils.isEmpty(id)){
                longs.add(Long.valueOf(id));
            }
        }
        map.put("ids", longs);
        int affectCount = sqlSessionTemplate.delete(getDeleteByIdsStatement(), map);
        log.debug("--------------> 删除数据 {} 条",affectCount);
    }

    /**
     * 查询所有的数据 调用分页的sql,去除分页的逻辑代码
     * @return
     */
    @Deprecated
    @Override
    public List findAll()  {
        return  sqlSessionTemplate.selectList(getPageStatementForAll(),null);
    }

    /**
     * 调用的xml 中的 <update></update> 标签
     * @param statementName sql 语句ID
     * @param parameter     参数
     * @return
     */
    @Override
    public int updateBySql(String statementName, Object parameter) {
        return sqlSessionTemplate.update(getNameSpace() +"."+statementName,parameter);
    }

    /**
     * 调用名空间的中<delete></delete> 标签
     * @param statementName 语句ID
     * @param parameter     参数
     * @return
     */
    @Override
    public int deleteBySql(String statementName, Object parameter) {
        return sqlSessionTemplate.delete(getNameSpace() +"."+statementName,parameter);
    }

    /**
     * 调用参数中的<insert></insert> 标签
     * @param statementName 语句ID
     * @param parameter     参数
     * @return
     */
    @Override
    public int insertBySql(String statementName, Object parameter) {
        return sqlSessionTemplate.insert(getNameSpace() +"."+statementName,parameter);
    }

    /**
     * 调用参数中的<select></select> 标签
     * @param statementName 语句ID
     * @param parameter     参数
     * @return 单个数据返回
     */
    @Override
    public E selectOne(String statementName, Object parameter)  {
        return sqlSessionTemplate.selectOne(getNameSpace() +"."+statementName,parameter);
    }

    /**
     * 调用参数中的<select></select> 标签
     * @param statementName 语句ID
     * @param parameter     参数
     * @return 多个数据
     */
    @Override
    public List selectList(String statementName, Object parameter) {
        return sqlSessionTemplate.selectList(getNameSpace() +"."+statementName,parameter);
    }

    /**
     * 调用参数中的<select></select> 标签
     * @param statementName 语句ID
     * @param parameter     参数
     * @return 单个数据返回
     */
    @Override
    public Long selectCount(String statementName, Object parameter)  {
        return sqlSessionTemplate.selectOne(getNameSpace() +"."+statementName,parameter);
    }


    /**
     * 获得对象的命名空间
     * @return 返回对象的对象的命名空间路径
     */
    protected String getNameSpace(){
        Type sType = getClass().getGenericSuperclass();
        Type[] generics = ((ParameterizedType) sType).getActualTypeArguments();
        Class<E> mTClass = (Class<E>) (generics[0]);
        return mTClass.getName()+"Mapper";
    }


    public String getInsertStatement() {
        return getNameSpace() + ".insert";
    }

    public String getBatchInsertStatement() {
        return getNameSpace() + ".inserts";
    }

    public String getUpdateStatement() {
        return getNameSpace() + ".updateById";
    }

    public String getDeleteStatement() {
        return getNameSpace() + ".deleteById";
    }

    public String getDeleteByIdsStatement() {
        return getNameSpace() + ".deleteByIds";
    }

    public String getPageStatementForPaging() {
        return getNameSpace()+  ".findPage";
    }
    public String getPageStatementForAll() {
        return getNameSpace()+  ".findAll";
    }
    public String getCountStatementForPaging() {
        return getNameSpace() + ".count";
    }

    public String getByIdStatement(){
        return getNameSpace() + ".getById";
    }

    private String getListByIdsStatement() {
        return getNameSpace()+".getByIdList";
    }
}
