package com.chinamcloud.spider.system.config;

import com.alibaba.druid.DbType;
import com.alibaba.druid.filter.Filter;
import com.alibaba.druid.filter.logging.Slf4jLogFilter;
import com.alibaba.druid.filter.stat.StatFilter;
import com.alibaba.druid.pool.DruidDataSource;
import com.alibaba.druid.util.StringUtils;
import com.chinamcloud.spider.system.config.interceptors.dm.DMKeywordInterceptor;
import com.chinamcloud.spider.system.config.typeHandlers.BitHex2BoolTypeHandler;
import com.chinamcloud.spider.system.config.interceptors.postgresql.PostgreSqlPrepareInterceptor;
import com.chinamcloud.spider.system.config.typeHandlers.KingbaseBitHexBoolTypeHandler;
import com.chinamcloud.spider.system.config.typeHandlers.PgLongTextTypeHandler;
import com.google.common.collect.Lists;
import lombok.Getter;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import org.apache.ibatis.mapping.VendorDatabaseIdProvider;
import org.apache.ibatis.plugin.Interceptor;
import org.apache.ibatis.session.SqlSessionFactory;
import org.apache.ibatis.type.TypeHandler;
import org.mybatis.spring.SqlSessionFactoryBean;
import org.mybatis.spring.SqlSessionTemplate;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.io.ClassPathResource;
import org.springframework.core.io.support.PathMatchingResourcePatternResolver;
import org.springframework.jdbc.datasource.DataSourceTransactionManager;
import org.springframework.transaction.annotation.EnableTransactionManagement;

import javax.sql.DataSource;
import java.sql.SQLException;
import java.util.List;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Created by jyy on 17/6/6.
 */
@Slf4j
@Configuration
//启用注解事务管理，使用CGLib代理
@EnableTransactionManagement(proxyTargetClass = true)
@Setter
@Getter
@ComponentScan(basePackages = "com.chinamcloud.spider")
public class DataSourceAutoConfiguration {

    //分布式环境各种加载环境配置
    @Value("${spring.druid.datasource.type}")
    private String type;

    @Value("${spring.druid.datasource.driverClassName}")
    private String driverClassName;

    @Value("${spring.druid.datasource.url}")
    private String url;

    @Value("${spring.druid.datasource.username}")
    private String username;

    @Value("${spring.druid.datasource.password}")
    private String password;

    @Value("${spring.druid.datasource.initialSize}")
    private Integer initialSize;

    @Value("${spring.druid.datasource.minIdle}")
    private Integer minIdle;

    @Value("${spring.druid.datasource.maxActive}")
    private Integer maxActive;

    @Value("${spring.druid.datasource.maxWait}")
    private Long maxWait;

    @Value("${spring.druid.datasource.timeBetweenEvictionRunsMillis}")
    private Long timeBetweenEvictionRunsMillis;

    @Value("${spring.druid.datasource.minEvictableIdleTimeMillis}")
    private Long minEvictableIdleTimeMillis;

    @Value("${spring.druid.datasource.validationQuery}")
    private String validationQuery;

    @Value("${spring.druid.datasource.testWhileIdle}")
    private boolean testWhileIdle;

    @Value("${spring.druid.datasource.testOnBorrow}")
    private boolean testOnBorrow;

    @Value("${spring.druid.datasource.testOnReturn}")
    private boolean testOnReturn;

    @Value("${spring.druid.datasource.poolPreparedStatements}")
    private boolean poolPreparedStatements;

    @Value("${spring.druid.datasource.maxPoolPreparedStatementPerConnectionSize}")
    private Integer maxPoolPreparedStatementPerConnectionSize;

    @Value("${spring.druid.datasource.filters}")
    private String filters;

    @Value("${spring.druid.datasource.connectionProperties}")
    private String connectionProperties;

    @Value("${spring.druid.datasource.useGlobalDataSourceStat:false}")
    private boolean useGlobalDataSourceStat;

    private String mapperPath = "**/*Mapper.xml";

    @Bean("spiderDruidDataSource")
    public DruidDataSource dataSource() throws Exception {
        DruidDataSource dataSource = new DruidDataSource();
        // 正则表达式
        Pattern pattern = Pattern.compile("jdbc:(\\w+)://.*?/([^?]+)");
        Matcher matcher = pattern.matcher(url);
        if (matcher.find()) {
            String jdbcType = matcher.group(1); // 提取 JDBC 类型
            String dbName = matcher.group(2); // 提取库名
            //驱动名
            String driverClassNameFinal = driverClassName;
            if (StringUtils.equalsIgnoreCase(jdbcType,DbType.dm.name())) {
                dataSource.setConnectionInitSqls(Lists.newArrayList("SET SCHEMA \"" + dbName +"\";"));
                if (StringUtils.isEmpty(driverClassName)) {
                    driverClassNameFinal = "dm.jdbc.driver.DmDriver";
                }
                dataSource.setDbType(DbType.dm);
            } else if(StringUtils.equalsIgnoreCase(jdbcType,DbType.kingbase.name()) || StringUtils.equalsIgnoreCase(jdbcType,"kingbase8")){
                System.out.println("当前数据库配置jdbcType="+jdbcType);
                //                dataSource.setConnectionInitSqls(Lists.newArrayList("SET SCHEMA " + dbName +";"));
                dataSource.setConnectionInitSqls(Lists.newArrayList("set names 'UTF-8';"));
                if (StringUtils.isEmpty(driverClassName)) {
                    driverClassNameFinal = "com.kingbase8.Driver";
                }
                dataSource.setDbType(DbType.kingbase);
            } else if(StringUtils.equalsIgnoreCase(jdbcType,DbType.postgresql.name())){
                System.out.println("当前数据库配置jdbcType="+jdbcType);
                dataSource.setConnectionInitSqls(Lists.newArrayList("set names 'UTF-8';"));
                if (StringUtils.isEmpty(driverClassName)) {
                    driverClassNameFinal = "org.postgresql.Driver";
                }
                dataSource.setDbType(DbType.postgresql);
            } else{
                dataSource.setConnectionInitSqls(Lists.newArrayList("set names utf8mb4;"));
                if (StringUtils.isEmpty(driverClassName)) {
                    driverClassNameFinal = "com.mysql.jdbc.Driver";
                }
                dataSource.setDbType(jdbcType);
            }
            dataSource.setDriverClassName(driverClassNameFinal);
        } else {
            System.out.println("Invalid JDBC URL." + ":" + url);
            throw new IllegalArgumentException("无效的jdbc连接信息:"+ url);
        }
        dataSource.setUrl(url);
        dataSource.setUsername(username);
        dataSource.setPassword(password);
        dataSource.setInitialSize(initialSize);
        dataSource.setMinIdle(minIdle);
        dataSource.setMaxActive(maxActive);
        dataSource.setMaxWait(maxWait);
        dataSource.setTimeBetweenEvictionRunsMillis(timeBetweenEvictionRunsMillis);
        dataSource.setMinEvictableIdleTimeMillis(minEvictableIdleTimeMillis);
        if (validationQuery != null && !"".equals(validationQuery)) {
            dataSource.setValidationQuery(validationQuery);
        }
        dataSource.setTestWhileIdle(testWhileIdle);
        dataSource.setTestOnBorrow(testOnBorrow);
        dataSource.setTestOnReturn(testOnReturn);
        dataSource.setMaxPoolPreparedStatementPerConnectionSize(maxPoolPreparedStatementPerConnectionSize);
        try {
            dataSource.setFilters(filters);//这是最关键的,否则SQL监控无法生效
            List<Filter> proxyFilters = dataSource.getProxyFilters();
            if (proxyFilters != null) {
                for (Filter proxyFilter : proxyFilters) {
                    if (proxyFilter instanceof StatFilter) {
                        ((StatFilter) proxyFilter).setMergeSql(false);
                    }
                }
            }
        } catch (SQLException e) {
            System.err.println("设置数据源过滤器出错:" + url);
        }
        if(connectionProperties != null && !"".equals(connectionProperties)){
            Properties connectProperties = new Properties();
            String[] propertiesList = connectionProperties.split(";");
            for(String propertiesTmp:propertiesList){
                String[] obj = propertiesTmp.split("=");
                String key = obj[0];
                String value = obj[1];
                connectProperties.put(key,value);
            }
            dataSource.setConnectProperties(connectProperties);
        }
        dataSource.getProxyFilters().add(slf4jLogFilter());
        dataSource.setUseGlobalDataSourceStat(useGlobalDataSourceStat);
        return dataSource;
    }

    @Bean(name = "spiderSqlSessionFactory")
    public SqlSessionFactory sqlSessionFactoryBean(@Qualifier("spiderDruidDataSource") final DruidDataSource druidDataSource) throws Exception {
        SqlSessionFactoryBean sqlSessionFactoryBean = new SqlSessionFactoryBean();
        sqlSessionFactoryBean.setDataSource(druidDataSource);
        PathMatchingResourcePatternResolver resolver = new PathMatchingResourcePatternResolver();
        sqlSessionFactoryBean.setMapperLocations(resolver.getResources("classpath*:"+mapperPath));
        sqlSessionFactoryBean.setConfigLocation(new ClassPathResource("mybatis-config.xml"));
        sqlSessionFactoryBean.setPlugins(createInterceptors(druidDataSource));
        sqlSessionFactoryBean.setDatabaseIdProvider(new VendorDatabaseIdProvider(){
            @Override
            public String getDatabaseId(DataSource dataSource) {
                return druidDataSource.getDbType();
            }
        });
        sqlSessionFactoryBean.setTypeHandlers(createTypeHandler(druidDataSource));
        return sqlSessionFactoryBean.getObject();
    }

    private static Interceptor[] createInterceptors(DruidDataSource dataSource) {
        if (dataSource != null) {
            String driverClassName = dataSource.getDriverClassName();
            if (driverClassName.contains("dm")) {
                log.info("加载了达梦数据库拦截器DMKeywordInterceptor!!!");
                return new Interceptor[]{new DMKeywordInterceptor()};
            }
            if (driverClassName.contains("postgresql")) {
                log.info("加载了postgresql数据库拦截器PostgresSqlInterceptor!!!");
                return new Interceptor[]{new PostgreSqlPrepareInterceptor()};
            }
            if (driverClassName.contains("kingbase")) {
                log.info("加载了人大金仓数据库拦截器KingbaseKeywordInterceptor!!!");
                return new Interceptor[]{/*new KingbaseBatchSaveInterceptor()*/};
            }
        }
        return new Interceptor[0];
    }

    private static TypeHandler<?>[] createTypeHandler(DruidDataSource dataSource) {
        if (dataSource != null) {
            String driverClassName = dataSource.getDriverClassName();
            if (driverClassName.contains("postgresql")) {
                log.info("加载了postgresql类型处理器!!!");
                return new TypeHandler[]{new BitHex2BoolTypeHandler(), new PgLongTextTypeHandler()};
            }
            if (driverClassName.contains("kingbase")) {
                log.info("加载了人大金仓类型处理器!!!");
                return new TypeHandler[]{new KingbaseBitHexBoolTypeHandler()};
            }
        }
        return new TypeHandler[0];
    }

    @Bean("spiderDataSourceTransactionManager")
    public DataSourceTransactionManager transactionManager(@Qualifier("spiderDruidDataSource") DruidDataSource druidDataSource) throws Exception {
        return new DataSourceTransactionManager(druidDataSource);
    }

    @Bean
    public Slf4jLogFilter  slf4jLogFilter(){
        Slf4jLogFilter slf4jLogFilter = new Slf4jLogFilter();
        slf4jLogFilter.setStatementExecutableSqlLogEnable(true);
        slf4jLogFilter.setConnectionLogEnabled(true);
        slf4jLogFilter.setResultSetLogErrorEnabled(true);
        slf4jLogFilter.setStatementLogEnabled(true);
        return slf4jLogFilter;
    }

    @Bean(name = "spiderSqlSessionTemplate")
    public SqlSessionTemplate SqlSessionTemplate(@Qualifier("spiderSqlSessionFactory") SqlSessionFactory sqlSessionFactory) throws Exception {
        return new SqlSessionTemplate(sqlSessionFactory);
    }

}
