package com.chinamcloud.spider.system.config.interceptors.dm;

import com.chinamcloud.spider.system.config.sqlwrapper.DynamicBoundSql;
import com.chinamcloud.spider.system.config.sqlwrapper.DynamicSqlSource;
import lombok.SneakyThrows;
import org.apache.ibatis.executor.Executor;
import org.apache.ibatis.mapping.*;
import org.apache.ibatis.plugin.*;
import org.apache.ibatis.session.Configuration;
import org.apache.ibatis.session.ResultHandler;
import org.apache.ibatis.session.RowBounds;
import org.apache.ibatis.type.JdbcType;

import java.lang.reflect.Field;
import java.util.*;

@Intercepts({@Signature(type = Executor.class, method = "update", args = {MappedStatement.class, Object.class}),
        @Signature(type = Executor.class, method = "query", args = {MappedStatement.class, Object.class, RowBounds.class, ResultHandler.class})})
public class DMKeywordInterceptor implements Interceptor {

    @Override
    public Object intercept(final Invocation invocation) throws Throwable {
        MappedStatement mappedStatement = (MappedStatement) invocation.getArgs()[0];
        Class<? extends MappedStatement> mappedStatementClass = mappedStatement.getClass();
        Field sqlSourceField = mappedStatementClass.getDeclaredField("sqlSource");
        sqlSourceField.setAccessible(true);
        final SqlSource sqlSource = (SqlSource) sqlSourceField.get(mappedStatement);
        sqlSourceField.set(mappedStatement, DynamicSqlSource.getDynamicSqlSource(sqlSource,invocation));
        return invocation.proceed();
    }

    @Override
    public Object plugin(Object target) {
        return Plugin.wrap(target, this);
    }

    @Override
    public void setProperties(Properties properties) {
        // 可以在这里设置拦截器的属性
    }

    public enum ProcessTokenEnum {
        /**
         * 对于向preparedStatement中传入null值参数其jdbcType必须要正确不能为空
         */

        SET_NULL_JDBC_TYPE {
            boolean processFlag = false;

            private final Field PARAMETERMAPPING_JDBC_FIELD;
            {
                try {
                    PARAMETERMAPPING_JDBC_FIELD = ParameterMapping.class.getDeclaredField("jdbcType");
                    PARAMETERMAPPING_JDBC_FIELD.setAccessible(true);
                } catch (NoSuchFieldException e) {
                    throw new RuntimeException(e);
                }
            }

            @Override
            public boolean shouldProcess(DynamicBoundSql originSql, Invocation invocation) {
                return true;
            }

            /**
             *
             * 这个值在mybatis-config.xml被设置为了java.sql.JDBCType#NULL,这里设置回默认的java.sql.JDBCType#OTHER
             */
            @Override
            @SneakyThrows
            public DynamicBoundSql process(DynamicBoundSql oldSql, Invocation invocation) {
                if(!processFlag){
                    MappedStatement mappedStatement = (MappedStatement) invocation.getArgs()[0];
                    Configuration configuration = mappedStatement.getConfiguration();
                    configuration.setJdbcTypeForNull(JdbcType.OTHER);
                    processFlag = true;
                }
                //遍历获取jdbcType为空的parameterMapping
                List<ParameterMapping> parameterMappings = oldSql.getParameterMappings();
                ResultMap baseResultMap = ProcessTokenEnum.getBaseResultMap(invocation);
                if (baseResultMap == null) {
                    return oldSql;
                }
                List<ResultMapping> resultMappings = baseResultMap.getResultMappings();
                HashMap<String, JdbcType> propertyJdbcTypeMap = new HashMap<>();
                for (ResultMapping resultMapping : resultMappings) {
                    String property = resultMapping.getProperty();
                    JdbcType jdbcType_act = resultMapping.getJdbcType();
                    propertyJdbcTypeMap.put(property,jdbcType_act);
                }
                for (ParameterMapping parameterMapping : parameterMappings) {
                    JdbcType jdbcType = parameterMapping.getJdbcType();
                    if (jdbcType == null) {
                        String property = parameterMapping.getProperty();
                        JdbcType type = propertyJdbcTypeMap.get(property);
                        if ( type != null) {
                            PARAMETERMAPPING_JDBC_FIELD.set(parameterMapping, type);
                        }
                    }
                }
                return oldSql;
            }
        },

        /**
         * 达梦的关键词之前在mysql作为表字段了,查询的时候加双引号
         */
        KEYWORD {
            //需要加引号处理的token
            final String[] targetToken = {"COMMENT","SYNONYM","SYSTEM","STAT","RULE","PAGE","MAP","JOB","DAILY","CATALOG","ROLE_ID","VALUE","TYPE"};

            @Override
            public boolean shouldProcess(DynamicBoundSql originSql, Invocation invocation) {
                String upperCaseSql = originSql.getSql().toUpperCase();
                for (String token : targetToken) {
                    if (upperCaseSql.contains(token)) {
                        return true;
                    }
                }
                return false;
            }

            @Override
            @SneakyThrows
            public DynamicBoundSql process(final DynamicBoundSql oldSql, Invocation invocation) {
                String processSql = oldSql.getSql();
                for (String s : targetToken) {
                    processSql = processSql.replaceAll("(?i)(?<=[\\s,.])+" + s + "(?=[\\s,])+", "\"" + "$0" + "\"");
                }
                oldSql.setSql(processSql);
                return oldSql;
            }
        },

        /**
         * mysql函数subdate达梦不支持
         * 可以替换为add_time - **INTERVAL** **'60'** **SECOND**
         * 但是暂时不做兼容直接替换为 *mapper.xml改为写法 now() > add_time - interval '60' SECOND
         */
        @Deprecated
        SUBDATE,

        /**
         * mysql中非聚合字段不存在于group by中是非标准sql写法,
         * 达梦不直接支持但是提供了兼容方式:在select后加入 \/*+ GROUP_OPT_FLAG(1)*\/
         */
        GROUP_BY {
            //兼容命令
            private final String COMPATIBILITY_IDENTIFICATION = "/*+ GROUP_OPT_FLAG(1)*/";

            @Override
            public boolean shouldProcess(DynamicBoundSql dynamicSqlBound, Invocation invocation) {
                String originSql2 = dynamicSqlBound.getSql().replace(" ", "");
                //转小写
                originSql2 = originSql2.toUpperCase();
                return originSql2.contains("GROUPBY") && !originSql2.contains(COMPATIBILITY_IDENTIFICATION.toUpperCase());
            }

            @Override
            public DynamicBoundSql process(DynamicBoundSql dynamicSqlBound, Invocation invocation) {
                //正则获取sql中第一个select或者SELECT 位置,在其后加入COMPATIBILITY_IDENTIFICATION
                String sqlNew = dynamicSqlBound.getSql().replaceFirst("(?i)(?<=SELECT)[^aA]*?(?=FROM)", " " + COMPATIBILITY_IDENTIFICATION + " " + "$0");
                dynamicSqlBound.setSql(sqlNew);
                return dynamicSqlBound;
            }
        },


        /**
         * mysql/group_concat--->dm/wm_concat
         */
        GROUP_CONCAT {
            @Override
            public boolean shouldProcess(DynamicBoundSql originSql, Invocation invocation) {
                String lowerSql = originSql.getSql().toUpperCase();
                //正则如果包含函数group_concat(.)则替换为wm_concat
                return lowerSql.contains("GROUP_CONCAT(");
            }

            @Override
            public DynamicBoundSql process(DynamicBoundSql oldSql, Invocation invocation) {
                String sql = oldSql.getSql();
                String funcOld = "(?i)" + this.name() + "\\(";
                String funcNew = "WM_CONCAT" + "\\(";
                sql = sql.replaceAll(funcOld, funcNew);
                oldSql.setSql(sql);
                return oldSql;
            }
        },

        /**
         * 只需要数据迁移后将主键从identity(1,1)改为AUTO_INCREMENT就能很好兼容
         * <p style="color:orange">注意xml中指定useGeneratedKeys后不可手动插入主键,否则实体中的主键属性会被数据库返回的AUTO_INCREMENT值覆盖掉</p>
         */
        @Deprecated
        RETURNING_ID,
        /**
         * 只需要数据迁移后将主键从identity(1,1)改为AUTO_INCREMENT就能很好兼容
         * <p style="color:orange">注意xml中指定useGeneratedKeys后不可手动插入主键,否则实体中的主键属性会被数据库返回的AUTO_INCREMENT值覆盖掉</p>
         */
        @Deprecated
        INSERT_PK;

        private volatile static Map<String, ResultMap> resultMaps;

        /**
         * 是否处理这个sql
         */
        public boolean shouldProcess(DynamicBoundSql originSql, Invocation invocation) {
            return false;
        }

        /**
         * 处理sql
         *
         * @param oldSql 老sql
         * @return 新sql
         */
        public DynamicBoundSql process(DynamicBoundSql oldSql, Invocation invocation) {
            return oldSql;
        }

        /**
         * 获取resultMap
         * @return 获取resultMap
         */
        @SneakyThrows
        @SuppressWarnings("unchecked")
        private static Map<String, ResultMap> getResultMaps(Invocation invocation) {
            if (resultMaps == null) {
                synchronized (DMKeywordInterceptor.class) {
                    if (resultMaps == null) {
                        MappedStatement arg0 = (MappedStatement) invocation.getArgs()[0];
                        Configuration configuration = arg0.getConfiguration();
                        Field resultMapField = Configuration.class.getDeclaredField("resultMaps");
                        resultMapField.setAccessible(true);
                        resultMaps = (Map<String, ResultMap>) resultMapField.get(configuration);
                    }
                }
            }
            return resultMaps;
        }

        private static ResultMap getBaseResultMap(Invocation invocation) {
            Object arg0 = invocation.getArgs()[0];
            if (arg0 instanceof MappedStatement) {
                MappedStatement statement = (MappedStatement) arg0;
                String id = statement.getId();
                //获取命名空间
                String mapperNameSpace = id.substring(0, id.lastIndexOf("."));
                //获取BaseResultMap
                Map<String, ResultMap> resultMaps = getResultMaps(invocation);
                ResultMap baseResultMap = null;
                try {
                    baseResultMap = resultMaps.get(mapperNameSpace + "." + "BaseResultMap");
                } catch (Exception e) {
                    //不存在baseResultMap的xml
                }
                return baseResultMap;
            }
            return null;
        }
    }
}
