/*
 * Decompiled with CFR 0.152.
 */
package io.lettuce.core.cluster.topology;

import io.lettuce.core.ConnectionFuture;
import io.lettuce.core.LettuceStrings;
import io.lettuce.core.RedisCommandInterruptedException;
import io.lettuce.core.RedisConnectionException;
import io.lettuce.core.RedisException;
import io.lettuce.core.RedisURI;
import io.lettuce.core.api.StatefulRedisConnection;
import io.lettuce.core.cluster.models.partitions.Partitions;
import io.lettuce.core.cluster.models.partitions.RedisClusterNode;
import io.lettuce.core.cluster.topology.AsyncConnections;
import io.lettuce.core.cluster.topology.Connections;
import io.lettuce.core.cluster.topology.NodeConnectionFactory;
import io.lettuce.core.cluster.topology.NodeTopologyView;
import io.lettuce.core.cluster.topology.NodeTopologyViews;
import io.lettuce.core.cluster.topology.RedisClusterNodeSnapshot;
import io.lettuce.core.cluster.topology.RefreshFutures;
import io.lettuce.core.cluster.topology.Requests;
import io.lettuce.core.cluster.topology.TimedAsyncCommand;
import io.lettuce.core.cluster.topology.TopologyComparators;
import io.lettuce.core.codec.StringCodec;
import io.lettuce.core.resource.ClientResources;
import io.netty.util.concurrent.EventExecutorGroup;
import io.netty.util.internal.logging.InternalLogger;
import io.netty.util.internal.logging.InternalLoggerFactory;
import java.net.SocketAddress;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;

public class ClusterTopologyRefresh {
    private static final InternalLogger logger = InternalLoggerFactory.getInstance(ClusterTopologyRefresh.class);
    private final NodeConnectionFactory nodeConnectionFactory;
    private final ClientResources clientResources;

    public ClusterTopologyRefresh(NodeConnectionFactory nodeConnectionFactory, ClientResources clientResources) {
        this.nodeConnectionFactory = nodeConnectionFactory;
        this.clientResources = clientResources;
    }

    public Map<RedisURI, Partitions> loadViews(Iterable<RedisURI> seed, Duration connectTimeout, boolean discovery) {
        if (!this.isEventLoopActive()) {
            return Collections.emptyMap();
        }
        long commandTimeoutNs = this.getCommandTimeoutNs(seed);
        Connections connections = null;
        try {
            Set<RedisURI> allKnownUris;
            Set<RedisURI> discoveredNodes;
            connections = this.getConnections(seed).get(commandTimeoutNs + connectTimeout.toNanos(), TimeUnit.NANOSECONDS);
            if (!this.isEventLoopActive()) {
                Map<RedisURI, Partitions> map = Collections.emptyMap();
                return map;
            }
            Requests requestedTopology = connections.requestTopology();
            Requests requestedClients = connections.requestClients();
            NodeTopologyViews nodeSpecificViews = this.getNodeSpecificViews(requestedTopology, requestedClients, commandTimeoutNs);
            if (discovery && this.isEventLoopActive() && !(discoveredNodes = ClusterTopologyRefresh.difference(allKnownUris = nodeSpecificViews.getClusterNodes(), this.toSet(seed))).isEmpty()) {
                Connections discoveredConnections = this.getConnections(discoveredNodes).optionalGet(commandTimeoutNs, TimeUnit.NANOSECONDS);
                connections = connections.mergeWith(discoveredConnections);
                if (this.isEventLoopActive()) {
                    requestedTopology = requestedTopology.mergeWith(discoveredConnections.requestTopology());
                    requestedClients = requestedClients.mergeWith(discoveredConnections.requestClients());
                    nodeSpecificViews = this.getNodeSpecificViews(requestedTopology, requestedClients, commandTimeoutNs);
                }
                if (nodeSpecificViews.isEmpty()) {
                    this.tryFail(requestedTopology, seed);
                }
                Map<RedisURI, Partitions> map = nodeSpecificViews.toMap();
                return map;
            }
            if (nodeSpecificViews.isEmpty()) {
                this.tryFail(requestedTopology, seed);
            }
            Map<RedisURI, Partitions> map = nodeSpecificViews.toMap();
            return map;
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new RedisCommandInterruptedException(e);
        }
        finally {
            if (connections != null) {
                try {
                    connections.close();
                }
                catch (Exception e) {
                    logger.debug("Cannot close ClusterTopologyRefresh connections", (Throwable)e);
                }
            }
        }
    }

    private void tryFail(Requests requestedTopology, Iterable<RedisURI> seed) {
        Throwable exception = null;
        for (RedisURI node : requestedTopology.nodes()) {
            TimedAsyncCommand<String, String, String> request = requestedTopology.getRequest(node);
            if (request == null || !request.isCompletedExceptionally()) continue;
            Throwable cause = RefreshFutures.getException(request);
            if (exception == null) {
                exception = new RedisException("Cannot retrieve initial cluster partitions from initial URIs " + seed, cause);
                continue;
            }
            if (cause == null) continue;
            exception.addSuppressed(cause);
        }
        if (exception != null) {
            throw exception;
        }
    }

    private Set<RedisURI> toSet(Iterable<RedisURI> seed) {
        return StreamSupport.stream(seed.spliterator(), false).collect(Collectors.toCollection(HashSet::new));
    }

    NodeTopologyViews getNodeSpecificViews(Requests requestedTopology, Requests requestedClients, long commandTimeoutNs) throws InterruptedException {
        ArrayList allNodes = new ArrayList();
        HashMap<String, Long> latencies = new HashMap<String, Long>();
        HashMap<String, Integer> clientCountByNodeId = new HashMap<String, Integer>();
        long waitTime = requestedTopology.await(commandTimeoutNs, TimeUnit.NANOSECONDS);
        requestedClients.await(commandTimeoutNs - waitTime, TimeUnit.NANOSECONDS);
        Set<RedisURI> nodes = requestedTopology.nodes();
        ArrayList<NodeTopologyView> views = new ArrayList<NodeTopologyView>();
        for (RedisURI nodeUri : nodes) {
            try {
                NodeTopologyView nodeTopologyView = NodeTopologyView.from(nodeUri, requestedTopology, requestedClients);
                if (!nodeTopologyView.isAvailable()) continue;
                RedisClusterNode node = nodeTopologyView.getOwnPartition();
                if (node.getUri() == null) {
                    node.setUri(nodeUri);
                } else {
                    node.addAlias(nodeUri);
                }
                ArrayList<RedisClusterNodeSnapshot> nodeWithStats = new ArrayList<RedisClusterNodeSnapshot>(nodeTopologyView.getPartitions().size());
                for (RedisClusterNode partition : nodeTopologyView.getPartitions()) {
                    if (!ClusterTopologyRefresh.validNode(partition)) continue;
                    RedisClusterNodeSnapshot redisClusterNodeSnapshot = new RedisClusterNodeSnapshot(partition);
                    nodeWithStats.add(redisClusterNodeSnapshot);
                    if (!partition.is(RedisClusterNode.NodeFlag.MYSELF)) continue;
                    latencies.put(partition.getNodeId(), nodeTopologyView.getLatency());
                    clientCountByNodeId.put(partition.getNodeId(), nodeTopologyView.getConnectedClients());
                }
                allNodes.addAll(nodeWithStats);
                Partitions partitions = new Partitions();
                partitions.addAll((Collection<? extends RedisClusterNode>)nodeWithStats);
                nodeTopologyView.setPartitions(partitions);
                views.add(nodeTopologyView);
            }
            catch (ExecutionException e) {
                logger.warn(String.format("Cannot retrieve partition view from %s, error: %s", nodeUri, e));
            }
        }
        for (RedisClusterNodeSnapshot node : allNodes) {
            node.setConnectedClients((Integer)clientCountByNodeId.get(node.getNodeId()));
            node.setLatencyNs((Long)latencies.get(node.getNodeId()));
        }
        TopologyComparators.SortAction sortAction = TopologyComparators.SortAction.getSortAction();
        for (NodeTopologyView view : views) {
            sortAction.sort(view.getPartitions());
            view.getPartitions().updateCache();
        }
        return new NodeTopologyViews(views);
    }

    private static boolean validNode(RedisClusterNode redisClusterNode) {
        if (redisClusterNode.is(RedisClusterNode.NodeFlag.NOADDR)) {
            return false;
        }
        return redisClusterNode.getUri() != null && redisClusterNode.getUri().getPort() != 0 && !LettuceStrings.isEmpty(redisClusterNode.getUri().getHost());
    }

    private AsyncConnections getConnections(Iterable<RedisURI> redisURIs) {
        AsyncConnections connections = new AsyncConnections();
        for (RedisURI redisURI : redisURIs) {
            if (redisURI.getHost() == null || connections.connectedNodes().contains(redisURI) || !this.isEventLoopActive()) continue;
            try {
                SocketAddress socketAddress = this.clientResources.socketAddressResolver().resolve(redisURI);
                ConnectionFuture<StatefulRedisConnection<String, String>> connectionFuture = this.nodeConnectionFactory.connectToNodeAsync(StringCodec.UTF8, socketAddress);
                CompletableFuture<StatefulRedisConnection<String, String>> sync = new CompletableFuture<StatefulRedisConnection<String, String>>();
                connectionFuture.whenComplete((connection, throwable) -> {
                    if (throwable != null) {
                        String message = String.format("Unable to connect to %s", socketAddress);
                        if (throwable instanceof RedisConnectionException) {
                            if (logger.isDebugEnabled()) {
                                logger.debug(throwable.getMessage(), throwable);
                            } else {
                                logger.warn(throwable.getMessage());
                            }
                        } else {
                            logger.warn(message, throwable);
                        }
                        sync.completeExceptionally(new RedisConnectionException(message, (Throwable)throwable));
                    } else {
                        connection.async().clientSetname("lettuce#ClusterTopologyRefresh");
                        sync.complete((StatefulRedisConnection<String, String>)connection);
                    }
                });
                connections.addConnection(redisURI, sync);
            }
            catch (RuntimeException e) {
                logger.warn(String.format("Unable to connect to %s", redisURI), (Throwable)e);
            }
        }
        return connections;
    }

    private boolean isEventLoopActive() {
        EventExecutorGroup eventExecutors = this.clientResources.eventExecutorGroup();
        return !eventExecutors.isShuttingDown();
    }

    public RedisURI getViewedBy(Map<RedisURI, Partitions> map, Partitions partitions) {
        for (Map.Entry<RedisURI, Partitions> entry : map.entrySet()) {
            if (entry.getValue() != partitions) continue;
            return entry.getKey();
        }
        return null;
    }

    private static <E> Set<E> difference(Set<E> set1, Set<E> set2) {
        HashSet result = new HashSet(set1.size());
        for (E e1 : set1) {
            if (set2.contains(e1)) continue;
            result.add(e1);
        }
        ArrayList<E> list = new ArrayList<E>(set2.size());
        for (E e : set2) {
            if (set1.contains(e)) continue;
            list.add(e);
        }
        result.addAll(list);
        return result;
    }

    private long getCommandTimeoutNs(Iterable<RedisURI> redisURIs) {
        RedisURI redisURI = redisURIs.iterator().next();
        return redisURI.getTimeout().toNanos();
    }
}

