/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.io.fs;

import java.io.BufferedWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.UncheckedIOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.CopyOption;
import java.nio.file.DirectoryNotEmptyException;
import java.nio.file.DirectoryStream;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.NotDirectoryException;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.Collections;
import java.util.HashSet;
import java.util.Objects;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.lang3.SystemUtils;
import org.neo4j.function.Predicates;
import org.neo4j.util.Preconditions;

public final class FileUtils {
    private static final int NUMBER_OF_RETRIES = 5;

    private FileUtils() {
        throw new AssertionError();
    }

    public static void delete(Path path) throws IOException {
        if (Files.isDirectory(path, new LinkOption[0])) {
            FileUtils.deleteDirectory(path);
        } else {
            FileUtils.deleteFile(path);
        }
    }

    public static void deleteDirectory(Path path) throws IOException {
        FileUtils.deleteDirectory(path, Predicates.alwaysTrue());
    }

    public static void deleteDirectory(Path path, Predicate<Path> removeFilePredicate) throws IOException {
        if (Files.notExists(path, new LinkOption[0])) {
            return;
        }
        if (!Files.isDirectory(path, new LinkOption[0])) {
            throw new NotDirectoryException(path.toString());
        }
        FileUtils.windowsSafeIOOperation(() -> Files.walkFileTree(path, new DeletingFileVisitor(removeFilePredicate)));
    }

    public static void deleteFile(Path file) throws IOException {
        if (Files.notExists(file, new LinkOption[0])) {
            return;
        }
        if (Files.isDirectory(file, new LinkOption[0]) && !FileUtils.isDirectoryEmpty(file)) {
            throw new DirectoryNotEmptyException(file.toString());
        }
        FileUtils.windowsSafeIOOperation(() -> Files.delete(file));
    }

    public static long blockSize(Path file) throws IOException {
        Path path;
        Objects.requireNonNull(file);
        for (path = file; path != null && !Files.exists(path, new LinkOption[0]); path = path.getParent()) {
        }
        if (path == null) {
            throw new IOException("Fail to determine block size for file: " + file);
        }
        return Files.getFileStore(path).getBlockSize();
    }

    public static void moveFile(Path toMove, Path target) throws IOException {
        if (Files.notExists(toMove, new LinkOption[0])) {
            throw new NoSuchFileException(toMove.toString());
        }
        if (Files.exists(target, new LinkOption[0])) {
            throw new FileAlreadyExistsException(target.toString());
        }
        try {
            Files.move(toMove, target, new CopyOption[0]);
        }
        catch (IOException e) {
            if (Files.isDirectory(toMove, new LinkOption[0])) {
                Files.createDirectories(target, new FileAttribute[0]);
                FileUtils.copyDirectory(toMove, target);
                FileUtils.deleteDirectory(toMove);
            }
            FileUtils.copyFile(toMove, target);
            FileUtils.deleteFile(toMove);
        }
    }

    public static Path moveFileToDirectory(Path toMove, Path targetDirectory) throws IOException {
        if (Files.notExists(targetDirectory, new LinkOption[0])) {
            Files.createDirectories(targetDirectory, new FileAttribute[0]);
        }
        if (!Files.isDirectory(targetDirectory, new LinkOption[0])) {
            throw new NotDirectoryException(targetDirectory.toString());
        }
        Path target = targetDirectory.resolve(toMove.getFileName());
        FileUtils.moveFile(toMove, target);
        return target;
    }

    public static void copyFileToDirectory(Path file, Path targetDirectory) throws IOException {
        if (Files.notExists(targetDirectory, new LinkOption[0])) {
            Files.createDirectories(targetDirectory, new FileAttribute[0]);
        }
        if (!Files.isDirectory(targetDirectory, new LinkOption[0])) {
            throw new NotDirectoryException(targetDirectory.toString());
        }
        Path target = targetDirectory.resolve(file.getFileName());
        FileUtils.copyFile(file, target);
    }

    public static void truncateFile(Path file, long position) throws IOException {
        try (FileChannel channel = FileChannel.open(file, StandardOpenOption.READ, StandardOpenOption.WRITE);){
            FileUtils.windowsSafeIOOperation(() -> channel.truncate(position));
        }
    }

    private static void waitAndThenTriggerGC() {
        try {
            Thread.sleep(500L);
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
        System.gc();
    }

    public static String fixSeparatorsInPath(String path) {
        String fileSeparator = System.getProperty("file.separator");
        if ("\\".equals(fileSeparator)) {
            path = path.replace('/', '\\');
        } else if ("/".equals(fileSeparator)) {
            path = path.replace('\\', '/');
        }
        return path;
    }

    public static void copyFile(Path srcFile, Path dstFile) throws IOException {
        FileUtils.copyFile(srcFile, dstFile, StandardCopyOption.REPLACE_EXISTING);
    }

    public static void copyFile(Path srcFile, Path dstFile, CopyOption ... copyOptions) throws IOException {
        Files.createDirectories(dstFile.getParent(), new FileAttribute[0]);
        Files.copy(srcFile, dstFile, copyOptions);
    }

    public static void copyDirectory(Path from, Path to) throws IOException {
        FileUtils.copyDirectory(from, to, Predicates.alwaysTrue());
    }

    public static void copyDirectory(Path from, Path to, Predicate<Path> filter) throws IOException {
        Objects.requireNonNull(from);
        Objects.requireNonNull(to);
        Preconditions.checkArgument((boolean)from.isAbsolute(), (String)"From directory must be absolute");
        Preconditions.checkArgument((boolean)to.isAbsolute(), (String)"To directory must be absolute");
        Preconditions.checkArgument((boolean)Files.isDirectory(from, new LinkOption[0]), (String)"From is not a directory");
        Preconditions.checkArgument((!from.normalize().equals(to.normalize()) ? 1 : 0) != 0, (String)"From and to directories are the same");
        if (Files.notExists(to.getParent(), new LinkOption[0])) {
            Files.createDirectories(to.getParent(), new FileAttribute[0]);
        }
        Files.walkFileTree(from, new CopyingFileVisitor(from, to, filter, StandardCopyOption.REPLACE_EXISTING, StandardCopyOption.COPY_ATTRIBUTES));
    }

    public static void writeToFile(Path target, String text, boolean append) throws IOException {
        if (Files.notExists(target, new LinkOption[0])) {
            Files.createDirectories(target.getParent(), new FileAttribute[0]);
            Files.createFile(target, new FileAttribute[0]);
        }
        try (BufferedWriter out = Files.newBufferedWriter(target, StandardCharsets.UTF_8, StandardOpenOption.APPEND);){
            out.write(text);
        }
    }

    public static PrintWriter newFilePrintWriter(Path file, Charset charset) throws IOException {
        return new PrintWriter(Files.newBufferedWriter(file, charset, StandardOpenOption.APPEND));
    }

    public static Path path(String root, String ... path) {
        return FileUtils.path(Path.of(root, new String[0]), path);
    }

    public static Path path(Path root, String ... path) {
        for (String part : path) {
            root = root.resolve(part);
        }
        return root;
    }

    public static Path pathToFileAfterMove(Path fromDir, Path toDir, Path fileToMove) {
        if (!fileToMove.startsWith(fromDir) || fileToMove.equals(fromDir)) {
            throw new IllegalArgumentException("File " + fileToMove + " is not a sub path to dir " + fromDir);
        }
        return toDir.resolve(fromDir.relativize(fileToMove));
    }

    public static long countFilesInDirectoryPath(Path dir) throws IOException {
        try (Stream<Path> listing = Files.list(dir);){
            long l = listing.count();
            return l;
        }
    }

    public static void windowsSafeIOOperation(Operation operation) throws IOException {
        IOException storedIoe = null;
        for (int i = 0; i < 5; ++i) {
            try {
                operation.perform();
                return;
            }
            catch (IOException e) {
                storedIoe = e;
                FileUtils.waitAndThenTriggerGC();
                continue;
            }
        }
        throw (IOException)Objects.requireNonNull(storedIoe);
    }

    public static Path getCanonicalFile(Path file) {
        try {
            return Files.exists(file, new LinkOption[0]) ? file.toRealPath(new LinkOption[0]).normalize() : file.normalize();
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    public static void writeAll(FileChannel channel, ByteBuffer src, long position) throws IOException {
        int bytesWritten;
        long filePosition = position;
        long expectedEndPosition = filePosition + (long)src.limit() - (long)src.position();
        while ((filePosition += (long)(bytesWritten = channel.write(src, filePosition))) < expectedEndPosition) {
            if (bytesWritten > 0) continue;
            throw new IOException("Unable to write to disk, reported bytes written was " + bytesWritten);
        }
    }

    public static void writeAll(FileChannel channel, ByteBuffer src) throws IOException {
        int bytesWritten;
        long bytesToWrite = src.limit() - src.position();
        while ((bytesToWrite -= (long)(bytesWritten = channel.write(src))) > 0L) {
            if (bytesWritten > 0) continue;
            throw new IOException("Unable to write to disk, reported bytes written was " + bytesWritten);
        }
    }

    public static String getFileStoreType(Path path) {
        try {
            return Files.getFileStore(path).type();
        }
        catch (IOException e) {
            return "Unknown file store type: " + e.getMessage();
        }
    }

    public static void tryForceDirectory(Path directory) throws IOException {
        if (Files.notExists(directory, new LinkOption[0])) {
            throw new NoSuchFileException(String.format("The directory %s does not exist!", directory.toAbsolutePath()));
        }
        if (!Files.isDirectory(directory, new LinkOption[0])) {
            throw new NotDirectoryException(String.format("The path %s must refer to a directory!", directory.toAbsolutePath()));
        }
        if (SystemUtils.IS_OS_WINDOWS) {
            return;
        }
        try (FileChannel directoryChannel = FileChannel.open(directory, Collections.singleton(StandardOpenOption.READ), new FileAttribute[0]);){
            directoryChannel.force(true);
        }
    }

    public static boolean isDirectoryEmpty(Path directory) throws IOException {
        try (DirectoryStream<Path> dirStream = Files.newDirectoryStream(directory);){
            boolean bl = !dirStream.iterator().hasNext();
            return bl;
        }
    }

    public static Path[] listPaths(Path dir) {
        Path[] pathArray;
        block8: {
            Stream<Path> list = Files.list(dir);
            try {
                pathArray = (Path[])list.toArray(Path[]::new);
                if (list == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (list != null) {
                        try {
                            list.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException ignored) {
                    return null;
                }
            }
            list.close();
        }
        return pathArray;
    }

    public static interface Operation {
        public void perform() throws IOException;
    }

    private static class CopyingFileVisitor
    extends SimpleFileVisitor<Path> {
        private final Path from;
        private final Path to;
        private final Predicate<Path> filter;
        private final CopyOption[] copyOption;
        private final Set<Path> copiedPathsInDestination = new HashSet<Path>();

        CopyingFileVisitor(Path from, Path to, Predicate<Path> filter, CopyOption ... copyOption) {
            this.from = from.normalize();
            this.to = to.normalize();
            this.filter = filter;
            this.copyOption = copyOption;
        }

        @Override
        public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
            if (!this.from.equals(dir) && !this.filter.test(dir)) {
                return FileVisitResult.SKIP_SUBTREE;
            }
            if (this.copiedPathsInDestination.contains(dir)) {
                return FileVisitResult.SKIP_SUBTREE;
            }
            Path target = this.to.resolve(this.from.relativize(dir));
            if (!Files.exists(target, new LinkOption[0])) {
                Files.createDirectory(target, new FileAttribute[0]);
                if (this.isInDestination(target)) {
                    this.copiedPathsInDestination.add(target);
                }
            }
            return FileVisitResult.CONTINUE;
        }

        @Override
        public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
            if (!this.filter.test(file)) {
                return FileVisitResult.CONTINUE;
            }
            if (!this.copiedPathsInDestination.contains(file)) {
                Path target = this.to.resolve(this.from.relativize(file));
                Files.copy(file, target, this.copyOption);
                if (this.isInDestination(target)) {
                    this.copiedPathsInDestination.add(target);
                }
            }
            return FileVisitResult.CONTINUE;
        }

        private boolean isInDestination(Path path) {
            return path.startsWith(this.to);
        }
    }

    private static class DeletingFileVisitor
    extends SimpleFileVisitor<Path> {
        private final Predicate<Path> removeFilePredicate;
        private int skippedFiles;

        DeletingFileVisitor(Predicate<Path> removeFilePredicate) {
            this.removeFilePredicate = removeFilePredicate;
        }

        @Override
        public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
            if (this.removeFilePredicate.test(file)) {
                Files.delete(file);
            } else {
                ++this.skippedFiles;
            }
            return FileVisitResult.CONTINUE;
        }

        @Override
        public FileVisitResult postVisitDirectory(Path dir, IOException e) throws IOException {
            if (e != null) {
                throw e;
            }
            try {
                if (this.skippedFiles == 0 || FileUtils.isDirectoryEmpty(dir)) {
                    Files.delete(dir);
                }
                return FileVisitResult.CONTINUE;
            }
            catch (DirectoryNotEmptyException notEmpty) {
                String reason = DeletingFileVisitor.notEmptyReason(dir, notEmpty);
                throw new IOException(notEmpty.getMessage() + ": " + reason, notEmpty);
            }
        }

        private static String notEmptyReason(Path dir, DirectoryNotEmptyException notEmpty) {
            String string;
            block8: {
                Stream<Path> list = Files.list(dir);
                try {
                    string = list.map(p -> String.valueOf(p.getFileName())).collect(Collectors.joining("', '", "'", "'."));
                    if (list == null) break block8;
                }
                catch (Throwable throwable) {
                    try {
                        if (list != null) {
                            try {
                                list.close();
                            }
                            catch (Throwable throwable2) {
                                throwable.addSuppressed(throwable2);
                            }
                        }
                        throw throwable;
                    }
                    catch (Exception e) {
                        notEmpty.addSuppressed(e);
                        return "(could not list directory: " + e.getMessage() + ")";
                    }
                }
                list.close();
            }
            return string;
        }
    }
}

